package com.xebialabs.deployit.engine.tasker

import akka.actor._
import akka.event.EventStream
import com.xebialabs.deployit.engine.api.execution.{FetchMode, TaskWithBlock}
import grizzled.slf4j.Logging

object TaskRegistryExtension extends ExtensionId[TaskRegistryExtension] with ExtensionIdProvider {
    //The lookup method is required by ExtensionIdProvider,
    // so we return ourselves here, this allows us
    // to configure our extension to be loaded when
    // the ActorSystem starts up
    override def lookup: ExtensionId[TaskRegistryExtension] = TaskRegistryExtension

    //This method will be called by Akka
    // to instantiate our Extension
    override def createExtension(system: ExtendedActorSystem) = new TaskRegistryExtension(system.eventStream)
}

class TaskRegistryExtension(eventStream: EventStream) extends Extension with Logging {
  info("Booting Task Registry")
  val store: Cache[String, Task] = Cache()

  def countTasks: Int = store.size

  def getTask(id: TaskId): Option[Task] = store.get(id)

  def getTasks(fetchMode: FetchMode): Seq[TaskWithBlock] = {
    val tasks = store.values.toList
    fetchMode match {
      case FetchMode.SUMMARY => tasks.map(t => new TaskSummary(t))
      case _ => tasks
    }
  }

  def deleteTask(id: TaskId): Unit =  {
    store.remove(id)
    if (store.isEmpty) {
      debug("Task registry empty.")
      eventStream.publish(TaskRegistryEmpty())
    }
  }

  protected[tasker] def clear(): Unit = store.clear()

  def register(task: Task) {
    if (store.contains(task.getId)) throw new IllegalStateException(s"Task with id [${task.getId}] is already registered")
    store.put(task.getId, task)
  }
}

case class TaskRegistryEmpty()