package com.xebialabs.deployit.engine.tasker

import akka.actor._
import com.xebialabs.deployit.engine.api.distribution.TaskExecutionWorkerRepository
import com.xebialabs.deployit.engine.tasker.TaskManagingActor.messages.Register
import com.xebialabs.deployit.engine.tasker.TaskReceiver.messages._
import com.xebialabs.deployit.engine.tasker.distribution.TaskDistributor.messages._
import com.xebialabs.deployit.engine.tasker.distribution.TasksManager
import com.xebialabs.deployit.engine.tasker.messages.{Enqueue, Registered}
import com.xebialabs.deployit.engine.tasker.repository.{ActiveTaskRepository, PendingTaskRepository}
import grizzled.slf4j.Logging

object TaskReceiver {
  def props(workerAddress: String, taskRepository: ActiveTaskRepository, pendingTaskRepository: PendingTaskRepository,
            workerRepository: TaskExecutionWorkerRepository, tasksManager: ActorRef): Props =
    Props(new TaskReceiver(workerAddress, taskRepository, pendingTaskRepository, workerRepository, tasksManager))

  val name = "task-receiver"

  object messages {

    case class ExecuteTask(taskId: TaskId)

  }

}

class TaskReceiver(workerAddress: String, activeTaskRepository: ActiveTaskRepository, pendingTaskRepository: PendingTaskRepository,
                   workerRepository: TaskExecutionWorkerRepository, tasksManager: ActorRef) extends Actor with Logging {

  val taskReceiver: ActorRef = self

  override def receive: Receive = doReceive(Map())

  def doReceive(specs: Map[String, TaskSpecification]): Receive = {
    case ExecuteTask(taskId) =>
      logger.info(s"Received ExecuteTask[$taskId] message.")
      extractTaskSpecification(taskId) { spec =>
        workerRepository.getWorkerByAddress(workerAddress) match {
          case Some(worker) =>
            tasksManager ! CreateTaskActor(spec, worker.id)
            context.become(doReceive(specs + (taskId -> spec)))
          case None =>
            logger.debug(s"Worker [$workerAddress] was not found.")
        }
      }

    case TaskActorCreated(taskActor) =>
      logger.debug(s"Received [${TaskActorCreated(taskActor)}] message.")
      taskActor ! Register()

    case Registered(taskId) =>
      logger.debug(s"Received [${Registered(taskId)}] message.")
      specs(taskId) match {
        case (spec) =>
          activeTaskRepository.store(taskId, spec, workerPath(taskId))
          pendingTaskRepository.delete(taskId)
          sender() ! Enqueue(taskId)
          context.become(doReceive(specs - taskId))
        case _ =>
      }
  }

  private def workerPath(taskId: TaskId) = ActorPath.fromString(s"$workerAddress/user/${TasksManager.name}/$taskId")

  def extractTaskSpecification(taskId: TaskId)(func: (TaskSpecification) => Unit): Unit = {
    pendingTaskRepository.task(taskId, loadFullSpec = true) match {
      case Some(pendingTask) =>
        pendingTask.spec match {
          case Some(taskSpecification) => func(taskSpecification)
          case None => onTaskNotFound(taskId)
        }
      case None => onTaskNotFound(taskId)
    }

    def onTaskNotFound(taskId: TaskId): Unit = {
      EnqueueTaskListener.startListeners()
      throw new TaskNotFoundException("pending tasks repository", taskId)
    }
  }
}
