package com.xebialabs.deployit.engine.tasker

import akka.actor.Status.Failure
import akka.actor._
import akka.event.LoggingReceive
import com.github.nscala_time.time.Imports._
import com.xebialabs.deployit.engine.api.execution.TaskExecutionState._
import com.xebialabs.deployit.engine.api.execution.{TaskExecutionState, BlockExecutionState => BlockIs}
import com.xebialabs.deployit.engine.tasker.ArchiveActor.messages.SendToArchive
import com.xebialabs.deployit.engine.tasker.BlockExecutingActor.{BlockDone, BlockStateChanged}
import com.xebialabs.deployit.engine.tasker.TaskManagingActor.messages._
import com.xebialabs.deployit.engine.tasker.log.{ExternalStepLogFactory, StepLogFactory}
import com.xebialabs.deployit.engine.tasker.messages._
import com.xebialabs.deployit.engine.tasker.satellite.log.TaskStepLogger
import org.joda.time.DateTime

import scala.concurrent.duration._

object TaskManagingActor {
  def props(task: Task, archiver: ActorRef, stepLogFactory: StepLogFactory): Props = Props(new TaskManagingActor(task, archiver, stepLogFactory)).withDispatcher(stateManagementDispatcher)

  def getMessages: messages.type = messages

  object messages {

    case class Register()

    case class Schedule(taskId: TaskId, scheduleAt: DateTime)

    case class ArchiveTask(taskId: TaskId, notificationActor: ActorRef)

    case class Cancel(taskId: TaskId, notificationActor: ActorRef, runMode: RunMode = RunMode.NORMAL)

    case class Recovered(task: Task)

  }

}

class TaskManagingActor(val task: Task, val archiver: ActorRef, stepLogFactory: StepLogFactory) extends BaseExecutionActor with ModifyStepsSupport with UpdateStateSupport {

  import context._

  case object ScheduledStart

  val taskId: String = task.getId

  private val blockFinder: BlockFinder = task.getBlock

  override def preStart(): Unit = {
    super.preStart()
    stepLogFactory match {
      case e: ExternalStepLogFactory =>
        createChild(Props(new TaskStepLogger(e)), "log")
      case _ =>
        // No external logs.
    }
  }

  override def receive: Receive = {
    logStateChange("receive")
    ReceiveWithMdc(task)(LoggingReceive.withLabel("receive") {
      case Register() =>
        info(s"Received [Register] message for task [$taskId]")
        registerStateListeners(task)
        val origSender = sender()
        updateStateAndNotify(PENDING, Some({
          case TaskStateEventHandled(`taskId`, _, PENDING) =>
            becomeWithMdc(pending)

            TaskRegistryExtension(system).register(task)
            origSender ! Registered(taskId)
        }))
      case Recovered(`task`) =>
        info(s"Received [Recovered] message for task [$taskId]")
        registerStateListeners(task)
        task.getState match {
          case PENDING | QUEUED =>
            becomeWithMdc(pending)
          case SCHEDULED if task.getScheduledDate.isAfterNow =>
            doSchedule(task.getScheduledDate, createOrLookupChildForTaskBlock())
          case SCHEDULED =>
            doEnqueue(createOrLookupChildForTaskBlock(), sendConfirmation = false)
          case EXECUTED =>
            becomeWithMdc(canBeArchived)
          case STOPPED | FAILED | ABORTED =>
            becomeWithMdc(readyForRestart(createOrLookupChildForTaskBlock()))
          case _ =>
            sender() ! Failure(new IllegalStateException(s"Cannot recover a task which is in state [${task.getState}]."))
        }
        TaskRegistryExtension(system).register(task)
        sender() ! Registered(taskId)
    })
  }

  def pending: Receive = {
    logStateChange("pending")
    LoggingReceive.withLabel("pending")(modifySteps orElse ReceiveOrFail("pending") {
      case Enqueue(`taskId`) =>
        doEnqueue(createOrLookupChildForTaskBlock())
      case Schedule(`taskId`, scheduleAt) =>
        doSchedule(scheduleAt, createOrLookupChildForTaskBlock())
      case Cancel(`taskId`, notificationActor, _) =>
        doCancelWhenPending(Some(notificationActor))
    })
  }

  def queued(blockActor: ActorRef): Receive = {
    logStateChange("queued")
    LoggingReceive.withLabel("queued")(ReceiveOrFail("queued") {
      case Start(`taskId`, _) =>
        doStart(blockActor)
    })
  }

  def scheduled(scheduleHandle: Cancellable, blockActor: ActorRef): Receive = {
    logStateChange("scheduled")
    LoggingReceive.withLabel("scheduled") {
      case Enqueue(`taskId`) =>
        scheduleHandle.cancel()
        doEnqueue(blockActor)
      case Schedule(`taskId`, scheduleAt) =>
        scheduleHandle.cancel()
        doSchedule(scheduleAt, blockActor)
      case ScheduledStart =>
        doEnqueue(blockActor, sendConfirmation = false)
      case Cancel(`taskId`, notificationActor, _) if task.getStartDate == null =>
        scheduleHandle.cancel()
        doCancelWhenPending(Some(notificationActor))
      case Cancel(`taskId`, notificationActor, runMode) if task.getStartDate != null =>
        scheduleHandle.cancel()
        doCancel(Some(notificationActor), runMode)
      case _: TaskStateEventHandled =>
      case msg: AnyRef =>
        throw new IllegalStateException(msg.toString)
    }
  }

  def awaitHandled(state: TaskExecutionState, nextBehaviour: Receive): Option[Receive] = Some({
    case TaskStateEventHandled(`taskId`, _, `state`) =>
      becomeWithMdc(nextBehaviour)
  })

  def executing(blockActor: ActorRef): Receive = {
    logStateChange("executing")
    LoggingReceive.withLabel("executing")(ReceiveOrFail("executing") {
      case m@BlockStateChanged(`taskId`, _, _, state) =>
        debug(s"executing(), Received $m")
        state match {
          case BlockIs.PENDING | BlockIs.EXECUTING =>
          case BlockIs.DONE =>
            updateStateAndNotify(EXECUTED, awaitHandled(EXECUTED, executed))
          case BlockIs.FAILING =>
            updateStateAndNotify(FAILING, None)
            becomeWithMdc(failing(blockActor, readyForRestart(blockActor)))
          case BlockIs.FAILED =>
            updateStateAndNotify(FAILED, awaitHandled(FAILED,
              failed(blockActor, forwardStepModification orElse readyForRestart(blockActor))))
          case BlockIs.STOPPING =>
            updateStateAndNotify(STOPPING, None)
            becomeWithMdc(stopping(blockActor, readyForRestart(blockActor)))
          case BlockIs.STOPPED =>
            updateStateAndNotify(STOPPED, awaitHandled(STOPPED,
              stopped(blockActor, forwardStepModification orElse readyForRestart(blockActor))))
          case BlockIs.ABORTING =>
            updateStateAndNotify(ABORTING, None)
            becomeWithMdc(aborting(blockActor, readyForRestart(blockActor)))
          case BlockIs.ABORTED =>
            updateStateAndNotify(ABORTED, awaitHandled(ABORTED,
              aborted(blockActor, forwardStepModification orElse readyForRestart(blockActor))))
        }
      case s@Stop(`taskId`) =>
        debug(s"Received [$s], now stopping execution")
        blockActor ! s
      case m@Abort(`taskId`) =>
        debug(s"Received [$m], now aborting execution")
        blockActor ! m
    })
  }

  def executed: Receive = {
    logStateChange("executed")
    LoggingReceive.withLabel("executed")(ReceiveOrFail("executed") {
      case m@BlockDone(`taskId`, _) =>
        debug(s"executed(), Received $m")
        task.recordCompletion()
        notifyTaskDone()
        becomeWithMdc(canBeArchived)
        unstashAll()
      case ArchiveTask(`taskId`, _) => stash()
    })
  }

  def canBeArchived: Receive = {
    logStateChange("canBeArchived")
    LoggingReceive.withLabel("canBeArchived")(ReceiveOrFail("canBeArchived") {
      case ArchiveTask(`taskId`, notificationActor) =>
        info(s"Received [ArchiveTask] message for task [$taskId]")
        updateStateAndNotify(DONE)
        becomeWithMdc(waitForStateHandledThenArchive(Set(DONE), Some(notificationActor)))
    })
  }

  def archiving(notificationActor: Option[ActorRef]): Receive = {
    logStateChange("archiving")
    LoggingReceive.withLabel("archiving") {
      case Archived(`taskId`) =>
        notifyTaskDone()
        task.getState match {
          case DONE => notificationActor.foreach { _ ! Archived(taskId) }
          case CANCELLED | FORCE_CANCELLED => notificationActor.foreach { _ ! Cancelled(taskId) }
          case _@state => warn(s"Unexpected state: $state for task: $taskId after task archive operation.")
        }
        TaskRegistryExtension(system).deleteTask(taskId)
        harakiri(s"Done with task [$taskId]")

      case fta@FailedToArchive(`taskId`, exception) =>
        info(s"Task [$taskId] failed to archive, going back to previous state.", exception)
        task.getState match {
          case DONE =>
            updateStateAndNotify(EXECUTED)
            becomeWithMdc(executed)
          case CANCELLED =>
            updateStateAndNotify(FAILED)
            val blockActor = createOrLookupChildForTaskBlock()
            becomeWithMdc(failed(blockActor, readyForRestart(blockActor)))
          case _@state => warn(s"Unexpected state: $state for task: $taskId when failed to archive.")
        }
        notifyTaskDone()
        notificationActor.foreach {
          _ ! fta
        }
      case _ => warn(s"Unexpected message during task: $taskId is being archived.")
    }
  }

  def failing(blockActor: ActorRef, restartState: => Receive, forceCancelledHandler: => Receive = null, runMode: RunMode = RunMode.NORMAL): Receive = {
    logStateChange("failing")
    LoggingReceive.withLabel("failing")(ReceiveOrFail("failing") {
      case m@BlockStateChanged(`taskId`, _, _, state) =>
        debug(s"failing(), received $m")
        state match {
          case BlockIs.PENDING | BlockIs.EXECUTING | BlockIs.DONE | BlockIs.FAILING | BlockIs.STOPPING | BlockIs.STOPPED =>
          case BlockIs.FAILED =>
            runMode match {
              case RunMode.NORMAL =>
                updateStateAndNotify(FAILED)
                becomeWithMdc(failed(blockActor, forwardStepModification orElse restartState, runMode))
              case RunMode.FORCE_CANCEL =>
                updateStateAndNotify(FORCE_CANCELLED)
                notifyTaskDone()
                becomeWithMdc(forceCancelledHandler)
            }
          case BlockIs.ABORTING =>
            updateStateAndNotify(ABORTING)
            becomeWithMdc(aborting(blockActor, restartState))
          case BlockIs.ABORTED =>
            updateStateAndNotify(ABORTED)
            becomeWithMdc(aborted(blockActor, forwardStepModification orElse restartState))
        }
      case s@Stop(`taskId`) =>
        debug(s"Received [$s], now stopping execution")
        blockActor ! s
      case m@Abort(`taskId`) =>
        debug(s"Received [$m], now aborting execution")
        blockActor ! m
    })
  }

  def failed(blockActor: ActorRef, restartState: => Receive, runMode: RunMode = RunMode.NORMAL): Receive = {
    logStateChange("failed")
    LoggingReceive.withLabel("failed")(ReceiveOrFail("failed") {
      case m@BlockDone(`taskId`, _) =>
        debug(s"failed(), received $m")
        task.recordFailure()
        task.recordCompletion()
        notifyTaskDone()
        becomeWithMdc(restartState)
        unstashAll()
      case Enqueue(`taskId`) | Schedule(`taskId`, _) | Cancel(`taskId`, _, _) => stash()
    })
  }

  def readyForRestart(blockActor: ActorRef): Receive = {
    logStateChange("readyForRestart")
    LoggingReceive.withLabel("readyForRestart")(modifySteps orElse ReceiveOrFail("readyForRestart") {
      case Enqueue(`taskId`) =>
        doEnqueue(blockActor)

      case Schedule(`taskId`, scheduleAt) =>
        doSchedule(scheduleAt, blockActor)

      case Cancel(`taskId`, notificationActor, runMode) =>
        doCancel(Some(notificationActor), runMode)
    })
  }

  def cancelling(blockActor: ActorRef, archiveAfterStateHandled: => Receive, readyState: => Receive, runMode: RunMode): Receive = {
    logStateChange("cancelling")
    LoggingReceive.withLabel("cancelling")(ReceiveOrFail("cancelling") {
      case m@BlockStateChanged(`taskId`, _, _, state) =>
        debug(s"cancelling(), received $m")
        state match {
          case BlockIs.PENDING | BlockIs.EXECUTING =>
          case BlockIs.DONE =>
            updateStateAndNotify(CANCELLED)
            notifyTaskDone()
            becomeWithMdc(archiveAfterStateHandled)
          case BlockIs.FAILING =>
            updateStateAndNotify(FAILING)
            becomeWithMdc(failing(blockActor, readyState, archiveAfterStateHandled, runMode))
          case BlockIs.FAILED =>
            runMode match {
              case RunMode.NORMAL =>
                updateStateAndNotify(FAILED)
                becomeWithMdc(failed(blockActor, forwardStepModification orElse readyState, runMode))
              case RunMode.FORCE_CANCEL =>
                updateStateAndNotify(FORCE_CANCELLED)
                notifyTaskDone()
                becomeWithMdc(archiveAfterStateHandled)
            }
          case BlockIs.STOPPING =>
            updateStateAndNotify(STOPPING)
            becomeWithMdc(stopping(blockActor, readyState))
          case BlockIs.STOPPED =>
            updateStateAndNotify(STOPPED)
            becomeWithMdc(stopped(blockActor, forwardStepModification orElse readyState))
          case BlockIs.ABORTING =>
            updateStateAndNotify(ABORTING)
            becomeWithMdc(aborting(blockActor, readyState))
          case BlockIs.ABORTED =>
            updateStateAndNotify(ABORTED)
            becomeWithMdc(aborted(blockActor, forwardStepModification orElse readyState))
        }
      case s@Stop(`taskId`) =>
        debug(s"Received [$s], now stopping execution")
        blockActor ! s
      case m@Abort(`taskId`) =>
        debug(s"Received [$m], now aborting execution")
        blockActor ! m
    })
  }

  def readyForCancelling(): Receive = {
    logStateChange("readyForCancelling")
    LoggingReceive.withLabel("readyForCancelling") {
      case Enqueue(`taskId`) =>
        doCancel(None, RunMode.NORMAL)
        sender() ! Enqueued
      case Cancel(`taskId`, nA, runMode) =>
        doCancel(Some(nA), runMode)
    }
  }

  def waitForStateHandledThenArchive(states: Set[TaskExecutionState], notificationActor: Option[ActorRef]): Receive = {
    logStateChange("waitForStateHandledThenArchive")
    LoggingReceive.withLabel("waitForStateHandledThenArchive") {
      case TaskStateEventHandled(`taskId`, _, state) if states.contains(state) && !task.getSpecification.isArchiveable =>
        becomeWithMdc(archiving(notificationActor))
        self ! Archived(task.getId)
      case TaskStateEventHandled(`taskId`, _, state) if states.contains(state) =>
        becomeWithMdc(archiving(notificationActor))
        archiver ! SendToArchive(task, self)
      case _@message =>
        debug(s"Unexpected message: {$message} when waiting for state handled before archive.")
    }
  }

  def stopping(blockActor: ActorRef, restartState: => Receive): Receive = {
    logStateChange("stopping")
    LoggingReceive.withLabel("stopping")(ReceiveOrFail("stopping") {
      case BlockStateChanged(`taskId`, _, _, state) =>
        state match {
          case BlockIs.PENDING | BlockIs.EXECUTING | BlockIs.STOPPING =>
          case BlockIs.DONE =>
            updateStateAndNotify(EXECUTED)
            becomeWithMdc(executed)
          case BlockIs.FAILING =>
            updateStateAndNotify(FAILING)
            becomeWithMdc(failing(blockActor, restartState))
          case BlockIs.FAILED =>
            updateStateAndNotify(FAILED)
            becomeWithMdc(failed(blockActor, forwardStepModification orElse restartState))
          case BlockIs.STOPPED =>
            updateStateAndNotify(STOPPED)
            becomeWithMdc(stopped(blockActor, forwardStepModification orElse restartState))
          case BlockIs.ABORTING =>
            updateStateAndNotify(ABORTING)
            becomeWithMdc(aborting(blockActor, restartState))
          case BlockIs.ABORTED =>
            updateStateAndNotify(ABORTED)
            becomeWithMdc(aborted(blockActor, forwardStepModification orElse restartState))
        }
      case s@Stop(`taskId`) =>
        debug(s"Received [$s], now stopping execution")
        blockActor ! s
      case m@Abort(`taskId`) =>
        debug(s"Received [$m], now aborting execution")
        blockActor ! m
    })
  }

  def stopped(blockActor: ActorRef, restartState: => Receive): Receive = {
    logStateChange("stopped")
    LoggingReceive.withLabel("stopped")(ReceiveOrFail("stopped") {
      case m@BlockDone(`taskId`, _) =>
        debug(s"stopped(), received $m")
        notifyTaskDone()
        becomeWithMdc(restartState)
        unstashAll()
      case Enqueue(`taskId`) | Schedule(`taskId`, _) | Cancel(`taskId`, _, _) => stash()
    })
  }

  def aborting(blockActor: ActorRef, restartState: => Receive): Receive = {
    logStateChange("aborting")
    LoggingReceive.withLabel("aborting")(ReceiveOrFail("aborting") {
      case BlockStateChanged(`taskId`, _, _, state) =>
        state match {
          case BlockIs.PENDING | BlockIs.EXECUTING | BlockIs.DONE | BlockIs.FAILING | BlockIs.STOPPING | BlockIs.STOPPED | BlockIs.FAILED | BlockIs.ABORTING =>
          case BlockIs.ABORTED =>
            updateStateAndNotify(ABORTED)
            becomeWithMdc(aborted(blockActor, forwardStepModification orElse restartState))
        }
    })
  }

  def aborted(blockActor: ActorRef, restartState: => Receive): Receive = {
    logStateChange("aborted")
    LoggingReceive.withLabel("aborted")(ReceiveOrFail("aborted") {
      case m@BlockDone(`taskId`, _) =>
        debug(s"aborted(), received $m")
        task.recordFailure()
        task.recordCompletion()
        notifyTaskDone()
        becomeWithMdc(restartState)
        unstashAll()
      case Enqueue(`taskId`) | Schedule(`taskId`, _) | Cancel(`taskId`, _, _) => stash()
    })
  }

  def calculateDelay(time: DateTime): FiniteDuration = FiniteDuration((DateTime.now to time).millis, MILLISECONDS)

  def doEnqueue(blockActor: ActorRef, sendConfirmation: Boolean = true) {
    val _sender = sender()
    info(s"Received [Enqueue] message for task [$taskId]")
    updateStateAndNotify(QUEUED, Some({
      case TaskStateEventHandled(`taskId`, _, QUEUED) =>
        if (sendConfirmation) {
          _sender ! Enqueued
        }
        becomeWithMdc(queued(blockActor))
        ActiveTasksQueue(system).addToQueueOrForward(self, Start(taskId))
    }))
  }

  def doStart(blockActor: ActorRef) {
    info(s"Received [Start] message for task [$taskId]")
    task.recordStart()
    updateStateAndNotify(EXECUTING, Some({
      case TaskStateEventHandled(`taskId`, _, EXECUTING) =>
        becomeWithMdc(executing(blockActor))
        info(s"sending ${Start(taskId)} to $blockActor")
        blockActor ! Start(taskId)
    }))

  }

  def doSchedule(scheduleAt: DateTime, blockActor: ActorRef) {
    info(s"Received [Schedule] message for task [$taskId]")
    task.setScheduledDate(scheduleAt)
    updateStateAndNotify(SCHEDULED)

    val delay: FiniteDuration = calculateDelay(scheduleAt)
    info(s"Going to schedule task [$taskId] at [${scheduleAt.toString("yyyy-MM-dd hh:mm:ss Z")}] which is [$delay] from now")

    if (delay.toMinutes < 0) {
      doEnqueue(blockActor, sendConfirmation = false)
    } else {
      val scheduleHandle: Cancellable = context.system.scheduler.scheduleOnce(delay, self, ScheduledStart)
      becomeWithMdc(scheduled(scheduleHandle, blockActor))
    }
  }

  def doCancelWhenPending(notificationActor: Option[ActorRef]) {
    info(s"Received [Cancel] message for task [$taskId]")
    updateStateAndNotify(CANCELLED)
    notifyTaskDone()
    TaskRegistryExtension(system).deleteTask(taskId)
    becomeWithMdc({
      case TaskStateEventHandled(`taskId`, _, CANCELLED) =>
        notificationActor.foreach {_ ! Cancelled(taskId)}
      case _@message =>
        warn(s"warn message: {$message} while waiting for cancelled state event")
    })
  }

  def doCancel(notificationActor: Option[ActorRef], runMode: RunMode): Unit = {
    info(s"Received [Cancel] message for task [$taskId]${if (runMode == RunMode.FORCE_CANCEL) " (with force)" else ""}")

    val blockActor = createOrLookupChildForTaskBlock()
    updateStateAndNotify(CANCELLING, Some({
      case TaskStateEventHandled(`taskId`, _, CANCELLING) =>
        val expectedStates = runMode match {
          case RunMode.NORMAL => Set(CANCELLED)
          case RunMode.FORCE_CANCEL => Set(CANCELLED, FORCE_CANCELLED)
        }
        val waitForStateHandled = waitForStateHandledThenArchive(expectedStates, notificationActor)
        val readyState = readyForCancelling()

        becomeWithMdc(cancelling(blockActor, waitForStateHandled, readyState, runMode))
        blockActor ! FinishUp(taskId, runMode)
    }))

  }

  private def forwardStepModification: Receive = forwardStepModification(taskId, createOrLookupChildForTaskBlock())

  private def modifySteps: Receive = modifySteps(taskId, blockFinder)

  def notifyTaskDone() {
    info(s"Task [$taskId] is completed with state [${task.getState}]")
    context.system.eventStream.publish(TaskDone(task))
  }

}
