package com.xebialabs.deployit.engine.tasker

import akka.actor.{ActorPath, ActorRef, ActorSelection, ActorSystem}
import akka.pattern._
import akka.util.Timeout
import com.xebialabs.deployit.engine.tasker.TaskManagingActor.messages.Register
import com.xebialabs.deployit.engine.tasker.distribution.WorkerManager.messages.NoWorkersError
import com.xebialabs.deployit.engine.tasker.distribution.{NoWorkersException, TasksManager}
import com.xebialabs.deployit.engine.tasker.messages.Registered
import com.xebialabs.deployit.engine.tasker.repository.{ActiveTaskRepository, PendingTaskRepository}
import com.xebialabs.xlplatform.settings.{CommonSettings, TaskerSettings}
import grizzled.slf4j.Logging

import scala.concurrent.{Await, Awaitable}
import scala.language.postfixOps
import scala.util.{Failure, Success}

trait TaskActorSupport extends Logging {

  protected val commonSettings: CommonSettings
  protected lazy val taskerSettings: TaskerSettings = commonSettings.tasker
  protected val distributor: ActorRef

  val taskRepository: ActiveTaskRepository
  val pendingTaskRepository: PendingTaskRepository
  val taskQueueService: TaskQueueService

  private[tasker] def getActorPathFromRepository(taskId: String): Option[ActorPath] = {
    taskRepository.workerAddress(taskId).map(wa => ActorPath.fromString(s"$wa/user/${TasksManager.name}/$taskId"))
  }

  private[tasker] def taskNotFoundException[T](taskId: TaskId) = new TaskNotFoundException("registry", taskId)

  private[tasker] def lookupTaskActor(taskid: String)(implicit system: ActorSystem): ActorSelection = {
    getActorPathFromRepository(taskid: String)
      .map(system.actorSelection)
      .getOrElse(throw taskNotFoundException(taskid))
  }

  private[tasker] def lookupTaskActorParent(taskid: String)(implicit system: ActorSystem): ActorSelection = {
    getActorPathFromRepository(taskid)
      .map(_ / "..")
      .map(system.actorSelection)
      .getOrElse(throw taskNotFoundException(taskid))
  }

  private[tasker] def askDistributor[T](msg: Any)(onTimeout: PartialFunction[Throwable, T]): T = {
    implicit val timeout: Timeout = taskerSettings.askTimeout
    try {
      Await.result(distributor ? msg, timeout.duration) match {
        case NoWorkersError =>
          throw NoWorkersException("There are currently no workers available, tasks cannot be executed. " +
            "Please contact your system administrator to correct the system.")
        case m: T => m
      }
    } catch onTimeout
  }

  private[tasker] def doRegister(taskActor: ActorRef, spec: TaskSpecification): String = {
    val taskId = askActor[Registered](taskActor, Register())(rethrowOnError).taskId
    taskRepository.store(taskId, spec, taskActor.path)
    taskId
  }

  private[tasker] def askActor[T](actor: ActorRef, msg: Any)(onTimeout: PartialFunction[Throwable, T]): T =
    doAsk { implicit t: Timeout => actor ? msg }(onTimeout)

  private[tasker] def askActor[T](actor: ActorSelection, msg: Any)(onTimeout: PartialFunction[Throwable, T]): T =
    doAsk { implicit t: Timeout => actor ? msg }(onTimeout)

  private[tasker] def doAsk[T](send: Timeout => Awaitable[Any])(onTimeout: PartialFunction[Throwable, T]): T = {
    val timeout: Timeout = taskerSettings.askTimeout
    try {
      Await.result(send(timeout), timeout.duration).asInstanceOf[T]
    } catch onTimeout
  }

  private[tasker] def prepareAndEnqueueTask(taskId: String): Unit = {
    pendingTaskRepository.prepareToEnqueue(taskId) match {
      case Success(_) =>
        taskQueueService.enqueueTask(taskId)
        logger.debug(s"Task [$taskId] enqueued.")
      case Failure(ex) =>
        logger.warn(s"Enqueue task [$taskId] failed due to the ${ex.getMessage}.")
    }
  }

  def rethrowOnError[T]: PartialFunction[Throwable, T] = {
    case t: Throwable => throw t
  }

}
