package com.xebialabs.deployit.engine.tasker.distribution.versioning

import com.typesafe.config.{Config, ConfigValue, ConfigValueType}

import scala.collection.JavaConverters._

class TypeSafeConfiguration(configuration: Config, whitelistedPaths: Seq[String], blacklistedPaths: Set[String]) extends ParentItem {

  def processItemValue(key: String, value: ConfigValue): Unit = {
    value.valueType() match {
      case ConfigValueType.NULL => addChild(new TypeSafeItem(key, "null"))
      case ConfigValueType.OBJECT => addChild(new TypeSafeItem(key, String.valueOf(value.unwrapped().hashCode()))) // if entrySet was used this won't happen
      case ConfigValueType.LIST => configuration.getList(key).forEach(processItemValue(key, _))
      case _ => addChild(new TypeSafeItem(key, value.render()))
    }
  }

  if(!configuration.isResolved) {
    configuration.resolve()
  }

  private val whitelisted = if (whitelistedPaths.isEmpty) Seq(configuration) else whitelistedPaths.map(configuration.withOnlyPath)

  private val blacklisted = blacklistedPaths.map {
    case s if s.endsWith(".") => new PathBlacklist(s)
    case s => new KeyBlacklist(s)
  }

  whitelisted.foreach(_.entrySet().asScala
    .filterNot(e => blacklisted.exists(_.matches(e.getKey)))
    .foreach(item => processItemValue(item.getKey, item.getValue)))

  sealed trait Blacklist {
    def matches(s: String): Boolean
  }

  class PathBlacklist(path: String) extends Blacklist {
    override def matches(s: String): Boolean = s.startsWith(path)
  }

  class KeyBlacklist(key: String) extends Blacklist {
    override def matches(s: String): Boolean = s.equals(key)
  }
}
