package com.xebialabs.deployit.engine.tasker

import akka.actor.{ActorRef, ActorSelection, ActorSystem}
import akka.util.Timeout
import com.xebialabs.deployit.engine.tasker.ArchivedListeningActor.Forward
import com.xebialabs.deployit.engine.tasker.TaskManagingActor.messages.{ArchiveTask, Cancel}
import com.xebialabs.deployit.engine.tasker.messages.RunMode
import com.xebialabs.deployit.engine.tasker.repository.CrudTaskRepository
import grizzled.slf4j.Logging

import scala.concurrent.duration.Duration
import scala.concurrent.{Await, Promise}
import scala.util.{Failure, Success}

object TaskFinalizer {
  def findActor(tasksManager: ActorRef, taskId: TaskId)(implicit system: ActorSystem): ActorSelection = {
    system.actorSelection(tasksManager.path / taskId)
  }
}

trait TaskFinalizer {
  def archive(actor: ActorSelection, taskId: TaskId): Unit
  def cancel(actor: ActorSelection, taskId: TaskId, runMode: RunMode = RunMode.NORMAL): Unit
}

class TaskFinalizerImpl(system: ActorSystem, taskRepository: CrudTaskRepository, implicit val askTimeout: Timeout) extends TaskFinalizer with Logging {
  override def archive(actor: ActorSelection, taskId: TaskId): Unit = {
    send(actor, taskId, ArchiveTask)
  }

  override def cancel(actor: ActorSelection, taskId: TaskId, runMode: RunMode = RunMode.NORMAL): Unit = {
    send(actor, taskId, Cancel(_, _, runMode))
  }

  private def send(actor: ActorSelection, taskid: TaskId, msg: (TaskId, ActorRef) => AnyRef): Unit = {
    val p = Promise[TaskId]()
    val listener: ActorRef = system.actorOf(ArchivedListeningActor.props(taskid, p))
    val message: AnyRef = msg(taskid, listener)

    listener ! Forward(actor, message)
    Await.ready(p.future, Duration.Inf)
    system.stop(listener)
    p.future.value.get match {
      case Failure(ex) => throw ex
      case Success(taskId) =>
        taskRepository.delete(taskid)
        debug(s"Task $taskId successfully archived.")
    }
  }
}
