package com.xebialabs.deployit.engine.tasker

import java.io.{File, FileFilter}
import java.nio.file.Files

import akka.actor._
import akka.pattern._
import akka.util.Timeout
import com.xebialabs.deployit.engine.spi.services.RepositoryFactory
import com.xebialabs.deployit.engine.tasker.RecoverySupervisorActor.{ReadAll, Tasks}
import com.xebialabs.deployit.engine.tasker.TaskFinalizer._
import com.xebialabs.deployit.engine.tasker.TaskManagingActor.messages.Recovered
import com.xebialabs.deployit.engine.tasker.distribution._
import com.xebialabs.deployit.engine.tasker.distribution.versioning.ConfigurationHashProvider
import com.xebialabs.deployit.engine.tasker.repository.CrudTaskRepository
import com.xebialabs.xlplatform.settings.TaskerSettings
import grizzled.slf4j.Logging

import scala.concurrent.Await
import scala.util.Try

object TaskExecutionWorker extends Logging {
  def initialize(masters: Seq[String],
                 hostnameToActorPathTransformer: String => String,
                 repositoryFactory: RepositoryFactory,
                 archive: Archive,
                 workerName: Option[String],
                 workerPublicKey: Array[Byte],
                 configurationHashProvider: ConfigurationHashProvider,
                 taskRepository: CrudTaskRepository,
                 taskerSettings: TaskerSettings,
                 shouldNotChunk : Boolean,
                 shutdown: () => Unit = shutdownSystem)(implicit system: ActorSystem): TaskRecoverer = {
    ActiveTasksQueue(system)

    val taskRegistry = TaskRegistryExtension(system)
    val archiver: ActorRef = system.actorOf(ArchiveActor.props(archive), ArchiveActor.name)
    val detectedWorkerName = workerName.getOrElse {
      val address = system.asInstanceOf[ExtendedActorSystem].provider.getDefaultAddress
      Seq(address.host, address.port).flatten.mkString(":")
    }
    val workerReceiver: ActorRef = system.actorOf(WorkerReceiver.props(masters, hostnameToActorPathTransformer.andThen(_ + WorkerManager.name), detectedWorkerName, workerPublicKey, configurationHashProvider.configurationHash, shutdown, () => taskRegistry.countTasks == 0), WorkerReceiver.name)
    val taskActorCreator: TaskActorCreator = new TaskActorCreatorImpl(repositoryFactory, archiver)
    val tasksManager: ActorRef = system.actorOf(TasksManager.props(taskActorCreator, taskRegistry, taskRepository, shouldNotChunk), TasksManager.name)
    val recoverySupervisor: ActorRef = system.actorOf(RecoverySupervisorActor.props(taskerSettings.recoveryDir), RecoverySupervisorActor.name)
    val taskRecoveryListener: ActorRef = system.actorOf(TaskRecoveryListener.props(recoverySupervisor), TaskRecoveryListener.name)
    val taskFinalizer: TaskFinalizer = new TaskFinalizerImpl(system, taskRepository, taskerSettings.askTimeout)
    val taskTransitioner: ActorRef = system.actorOf(TaskTransitionActor.props(taskFinalizer, tasksManager), TaskTransitionActor.name)

    new TaskRecoverer(taskerSettings, recoverySupervisor, tasksManager, taskFinalizer)
  }

  private def shutdownSystem(): Unit = {
    System.exit(0)
  }

  val inProcessWorkerName = "In-process worker"
  val localMasterAddress = s"/user/"

  def local(repositoryFactory: RepositoryFactory, archive: Archive, configurationHashProvider: ConfigurationHashProvider, taskRepository: CrudTaskRepository, taskerSettings: TaskerSettings)(implicit system: ActorSystem): TaskRecoverer =
    initialize(Seq(localMasterAddress), Predef.identity, repositoryFactory, archive, Some(inProcessWorkerName), null, configurationHashProvider, taskRepository, taskerSettings, shouldNotChunk = true, () => {warn("Attempt to shutdown local worker.")})

  class TaskRecoverer(taskerSettings: TaskerSettings,
                      recoverySupervisor: ActorRef,
                      tasksManager: ActorRef,
                      taskFinalizer: TaskFinalizer)
                     (implicit system: ActorSystem) {
    def recoverTasks(): Unit = {
      cleanUpTmpFiles()
      try {
        implicit val timeout: Timeout = taskerSettings.askTimeout
        val tasks = Await.result(recoverySupervisor ? ReadAll, timeout.duration).asInstanceOf[Tasks].tasks
        for (task <- tasks) Try(onRecover(task)).recover {
          case e: Exception => error("Error while recovering tasks.", e)
        }
      } catch {
        case e: Exception => error("Error while recovering.", e)
      }
    }

    private def cleanUpTmpFiles(): Unit = {
      val tmpFiles = taskerSettings.recoveryDir.listFiles(new FileFilter {
        override def accept(pathname: File): Boolean = pathname.getName.endsWith(".task.tmp")
      })
      Option(tmpFiles).toList.flatten.foreach { file =>
        try {
          warn(s"Deleting corrupted task file: ${file.getAbsolutePath}")
          Files.delete(file.toPath)
        } catch {
          case e: Throwable => error(s"Unable to delete corrupted recovery file: ${file.getAbsolutePath}", e)
        }
      }
    }

    def onRecover(task: Task): Unit = {
      implicit val timeout: Timeout = taskerSettings.askTimeout
      Await.ready(tasksManager ? Recovered(task), timeout.duration)
      if (task.cancelling) {
        taskFinalizer.cancel(findActor(tasksManager, task.getId), task.getId)
      }
    }
  }
}
