package com.xebialabs.deployit.engine.tasker

import java.io.{PrintWriter, Serializable, StringWriter}
import java.util

import com.xebialabs.deployit.plugin.api.flow.{ExecutionContext, ITask}
import com.xebialabs.deployit.plugin.api.inspection.InspectionContext
import com.xebialabs.deployit.plugin.api.services.Repository
import grizzled.slf4j.Logger

class StepExecutionContext(taskExecutionContext: TaskExecutionContext, step: TaskStep, task: Task, spec: TaskSpecification) extends ExecutionContext with Serializable {

  val ERROR_PREFIX = "[ERROR]: "

  val stepLogger = Logger.apply(step.getImplementation.getClass)

  override def setAttribute(name: String, value: scala.AnyRef) {
    taskExecutionContext.setAttribute(name, value)
  }

  override def logOutput(output: String) {
    logOutputRaw(output + "\n")
  }

  override def logOutputRaw(output: String): Unit = {
    stepLogger.info(output.stripSuffix("\n"))
    step.logBuilder.append(output)
    step.touch()
  }

  override def logError(error: String): Unit = {
    logErrorRaw(error + "\n")
  }

  override def logErrorRaw(error: String): Unit = {
    logErrorRaw(error, None)
  }

  def logError(error: String, t: Throwable): Unit = {
    logErrorRaw(error + "\n", Option(t))
  }

  private def logErrorRaw(error: String, t: Option[Throwable]): Unit = {
    if (t.isDefined) {
      stepLogger.error(error.stripSuffix("\n"), t.get)
    } else {
      stepLogger.error(error.stripSuffix("\n"))
    }
    appendErrorToLogRaw(error)
    t.foreach(logException)
    step.touch()
  }

  private def logException(t: Throwable): Unit = {
    val stringWriter: StringWriter = new StringWriter
    t.printStackTrace(new PrintWriter(stringWriter))
    stringWriter.toString.split("\n").foreach(s => step.logBuilder.append(ERROR_PREFIX).append(s).append("\n"))
  }

  private def appendErrorToLogRaw(error: String) {
    step.logBuilder.append(ERROR_PREFIX).append(error)
  }

  override def getAttribute(name: String): AnyRef = taskExecutionContext.getAttributes.getOrElse(name, null) match {
    case l: OldExecutionContextListenerCleanupTrigger => l.getWrappedListener
    case l@_ => l
  }

  override def getInspectionContext: InspectionContext = spec.getInspectionContext

  override def getRepository: Repository = taskExecutionContext.getRepository

  override def getTask: ITask = new ITask {

    override def getMetadata: util.Map[String, String] = task.getMetadata

    override def getUsername: String = task.getOwner

    override def getId: String = task.getId
  }

}
