package com.xebialabs.deployit.engine.tasker

import java.util.UUID

import akka.actor.Status.Failure
import akka.actor.{Actor, ActorRef, Cancellable, Props, Stash}
import akka.dispatch.MessageDispatcher
import akka.event.LoggingReceive.withLabel
import akka.pattern.pipe
import com.xebialabs.deployit.engine.api.execution.StepExecutionState._
import com.xebialabs.deployit.engine.api.execution.{StepExecutionState, TaskState}
import com.xebialabs.deployit.engine.tasker.MdcUtils.{mdcWithStep, mdcWithTask, withMdc}
import com.xebialabs.deployit.engine.tasker.StateChangeEventListenerActor.StepStateEvent
import com.xebialabs.deployit.engine.tasker.StepExecutingActor.internalMessages.{ExecutionStarted, StepTimedOut}
import com.xebialabs.deployit.engine.tasker.StepExecutingActor.messages._
import com.xebialabs.deployit.engine.tasker.messages.{Abort, StepStateEventHandled}
import com.xebialabs.deployit.plugin.api.flow.StepExitCode._
import com.xebialabs.deployit.plugin.api.flow.{ExecutionContext, Step, StepExitCode}
import com.xebialabs.deployit.repository.WorkDirContext
import com.xebialabs.xlplatform.settings.{CommonSettings, TaskerSettings}
import grizzled.slf4j.Logging
import org.springframework.security.core.context.SecurityContextHolder

import scala.concurrent.Future
import scala.concurrent.duration.FiniteDuration

object StepExecutingActor {
  def props(task: TaskState, ctx: TaskExecutionContext): Props =
    Props(classOf[StepExecutingActor], task, ctx).withDispatcher(stateManagementDispatcher)

  object messages {

    case class ExecuteStep(taskStep: TaskStep, executionCtx: ExecutionContext)

    case class RetryStep(taskStep: TaskStep, executionContext: ExecutionContext, originalSender: ActorRef)

    case class ExecutedStep(taskStep: TaskStep, executionContext: ExecutionContext, stepExitCode: StepExitCode, originalSender: ActorRef)

    case class FailedStep(taskStep: TaskStep, executionContext: ExecutionContext, t: Throwable, originalSender: ActorRef)

    case object StopRequested

  }

  object internalMessages {

    case class ExecutionStarted(taskStep: TaskStep)

    case object StepTimedOut
  }

}

class StepExecutingActor(val task: Task, ntec: TaskExecutionContext) extends Actor with Logging with BecomeWithMdc with Stash {
  val settings: TaskerSettings = CommonSettings(context.system).tasker
  val taskId: String = task.getId

  def receive: Receive = ReceiveWithMdc(task)(receiveExecuteStep())

  def receiveExecuteStep(stopRequested: Boolean = false, retryHandle: Option[(TaskStep, Cancellable)] = None): Receive = {

    case ExecuteStep(s, _) if s.getState == StepExecutionState.QUEUED || s.getState == StepExecutionState.EXECUTING =>
      sender() ! Failure(new IllegalStateException(s"The step [$s] is still either queued or executing, cannot run again."))

    case ExecuteStep(s, _) if s.getState.isFinal =>
      val implementation = s.getImplementation
      debug(s"Will not execute: $implementation with description: ${implementation.getDescription} because it has state: ${s.getState}")
      sender ! s.getState

    case ExecuteStep(s, _) if s.isMarkedForSkip =>
      s.recordStart()
      s.recordCompletion()
      setStepState(s, SKIPPED, sender())

    case message@ExecuteStep(s, ctx) =>
      debug(s"Got ExecuteStep message: $message")
      setStepState(s, StepExecutionState.QUEUED, sender())
      s.clearLog()
      info(s"Queued ${s.getImplementation} for execution")
      executeStep(s, s.getImplementation, ctx)

    case message@RetryStep(s, _, _) if retryHandle.isEmpty =>
      debug(s"Got RetryStep: $message, but the retryHandle was not set, checking step state")
      if (s.getState != EXECUTING) {
        debug("Ignoring RetryStep, the step is no longer executing")
      } else {
        sender() ! Failure(new IllegalStateException(s"Cannot Retry step $s when the retry handle is not set"))
      }

    case message@RetryStep(s, _, originalSender) if stopRequested =>
      debug(s"Got RetryStep: $message, but stop has been requested")
      setStepState(s, PAUSED, originalSender)

    case message@RetryStep(s, ctx, originalSender) =>
      debug(s"Got RetryStep: $message")
      info(s"Continuing ${s.getImplementation}")
      executeStep(s, s.getImplementation, ctx, originalSender, retry = true)

    case Abort(`taskId`) if retryHandle.isDefined =>
      debug("Received Abort while waiting to retry step.")
      // Cancel the retry handle
      val s = sender()
      retryHandle.foreach({ case (step, cancellable) =>
        debug("Cancelling retry message")
        cancellable.cancel()
        debug(s"Changing $step state to PAUSED")
        setStepState(step, PAUSED, s)
      })

    case StopRequested =>
      debug("Stop has been requested while waiting for an execution message")
      becomeWithMdc(receiveExecuteStep(stopRequested = true))
  }

  def executeStep(step: TaskStep, implStep: Step, ctx: ExecutionContext,
                          originalSender: ActorRef = sender(), retry: Boolean = false) {
    becomeWithMdc(withLabel("executionStartedOrDoneOrStashAbort")(
      receiveExecutionStarted orElse receiveExecutionDone() orElse receiveStashAbortStop)
    )
    // Never close over `self`.
    val me = self
    val f = Future {
      withMdc(mdcWithTask(task) ++ mdcWithStep(implStep)) {
        try {
          if (!retry) {
            info(s"Started ${step.getImplementation}")
            step.recordStart()
          }
          setStepState(step, StepExecutionState.EXECUTING, sender)
          step.registerCurrentThread()
          WorkDirContext.setWorkDir(task.getWorkDir)
          SecurityContextHolder.getContext.setAuthentication(task.getAuthentication)
          me ! ExecutionStarted(step)
          val result: StepExitCode = implStep.execute(ctx)
          step.recordCompletion()
          val executedStep = ExecutedStep(step, ctx, result, originalSender)
          executedStep
        } catch {
          case e: Exception =>
            FailedStep(step, ctx, e, originalSender)
        } finally {
          cleanup(step)
        }
      }
    }(context.system.dispatchers.lookup(stepDispatcher))

    implicit val executionContext: MessageDispatcher = context.system.dispatchers.lookup(stateManagementDispatcher)
    f pipeTo self
  }

  def receiveStopRequested(step: TaskStep, timeout: Cancellable): Receive = {
    case StopRequested => becomeWithMdc(withLabel("abortOrExecutionDoneWhenStopRequested")(
      receiveExecutionDone(stopRequested = true, Some(timeout)) orElse receiveAbortOrTimeout(step, timeout)))
  }

  def receiveExecutionDone(stopRequested: Boolean = false, timeout: Option[Cancellable] = None): Receive = {
    case message@ExecutedStep(step, ctx, result, originalSender) =>
      debug(s"Got ExecutedStep: $message")
      timeout.foreach(_.cancel())
      result match {
        case SUCCESS =>
          setStepState(step, DONE, originalSender)
        case FAIL =>
          setStepState(step, FAILED, originalSender)
        case PAUSE =>
          setStepState(step, PAUSED, originalSender)
        case RETRY if stopRequested =>
          debug("Retry while stop requested, will pause...")
          setStepState(step, PAUSED, originalSender)
        case RETRY =>
          val delay: FiniteDuration = settings.stepRetryDelay
          ctx.logOutput(s"Retrying in ${delay.toSeconds} seconds...")
          val retryStep = RetryStep(step, ctx, originalSender)
          val cancelHandle = context.system.scheduler.scheduleOnce(delay, self, retryStep)(context.dispatcher)
          becomeWithMdc(withLabel("retryStep")(receiveExecuteStep(retryHandle = Some((step, cancelHandle)))))
      }
    case FailedStep(step, ctx, t: Throwable, originalSender) =>
      ctx.logError("Step failed", t)
      timeout.foreach(_.cancel())
      step.recordCompletion()
      setStepState(step, FAILED, originalSender)
  }


  def doAbort(step: TaskStep) {
    info(s"[${step.getImplementation.getDescription}] : Received [Abort] message")
    step.interruptRunner()
    becomeWithMdc(withLabel("ExecutionDoneWhenAborted")(receiveExecutionDone(stopRequested = true)))
  }

  def receiveAbortOrTimeout(step: TaskStep, timeout: Cancellable): Receive = {
    case Abort(`taskId`) =>
      timeout.cancel()
      doAbort(step)
    case StepTimedOut =>
      warn(s"Step ${step.getDescription} timed out after ${settings.stepRunTimeout} - aborting!")
      doAbort(step)
  }

  def receiveExecutionStarted: Receive = {
    case ExecutionStarted(step) =>
      val timeout = context.system.scheduler.scheduleOnce(settings.stepRunTimeout, self, StepTimedOut)(executor = context.dispatcher)
      becomeWithMdc(withLabel("abortOrStopOrExecutionDone")(receiveAbortOrTimeout(step, timeout) orElse
        receiveStopRequested(step, timeout) orElse receiveExecutionDone(timeout = Some(timeout))
      ))
      unstashAll()
  }

  def receiveStashAbortStop: Receive = {
    case Abort(`taskId`) =>
      stash()
    case StopRequested =>
      stash()
  }


  private def cleanup(step: TaskStep): Unit = {
    debug(s"Doing cleanup for task $task")
    WorkDirContext.clear()
    SecurityContextHolder.getContext.setAuthentication(null)
    TaskStep.logger.info(step.getState.name)
  }

  private def setStepState(step: TaskStep, state: StepExecutionState, origSender: ActorRef) {
    val oldState: StepExecutionState = step.getState
    step.setState(state)
    val stepId: String = UUID.randomUUID().toString
    state match {
      case EXECUTING | QUEUED =>
        context.system.eventStream.publish(StepStateEvent(task.getId, stepId, task, step, oldState, state, Some(ntec), None))
      case _ =>
        becomeWithMdc(withLabel("receiveStepStateEventHandled")(receiveStepStateEventHandled(stepId, oldState, state, origSender)))
        context.system.eventStream.publish(StepStateEvent(task.getId, stepId, task, step, oldState, state, Some(ntec), Some(self)))
    }
  }


  def receiveStepStateEventHandled(stepId: String, oldState: StepExecutionState, newState: StepExecutionState,
                                   blockActor: ActorRef): Receive = {
    case StepStateEventHandled(`taskId`, `stepId`, `oldState`, `newState`) =>
      debug(s"Handled step state change [$oldState->$newState]")
      becomeWithMdc(withLabel("executeStep")(receiveExecuteStep()))
      unstashAll()
      blockActor ! newState
    case StepStateEventHandled(`taskId`, `stepId`, f, t) =>
      debug(s"Received step state change handled for $f -> $t, but was expecting $oldState -> $newState")
    case StepStateEventHandled(_, _, _, _) =>
    // Ignore, not for me
    case m@_ =>
      debug(s"Stashing $m")
      stash()
  }
}
