package com.xebialabs.deployit.engine.tasker.messages

import com.xebialabs.deployit.engine.api.execution.{BlockExecutionState, StepExecutionState, TaskExecutionState}
import com.xebialabs.deployit.engine.tasker._

case class Start(taskId: TaskId)
case class Enqueue(taskId: TaskId)
case class TaskDone(task: Task)
case class TaskStateEventHandled(taskId: TaskId, oldState: TaskExecutionState, newState: TaskExecutionState)
case class StepStateEventHandled(taskId: TaskId, stepId: StepId, oldState: StepExecutionState, newState: StepExecutionState)
case class Archived(taskId: TaskId)
case class FailedToArchive(taskId: TaskId, exception: Throwable)
case class Cancelled(taskId: TaskId)
case class Registered(taskId: TaskId)

case class Stop(taskId: TaskId)
case class Abort(taskId: TaskId)
case class FinishUp(taskId: TaskId)

case class StartBlock(taskId:TaskId, blockPath:BlockPath)
case class StopBlock(taskId:TaskId, blockPath:BlockPath)
case class AbortBlock(taskId:TaskId, blockPath:BlockPath)
case class RemoteTaskStateEvent(taskId: TaskId, oldState: TaskExecutionState, newState: TaskExecutionState)

sealed trait ModifySteps {
  val taskId: TaskId
  val blockPaths: Seq[BlockPath]
}

case class AddPauseStep(taskId: TaskId, blockPath: BlockPath) extends ModifySteps {
  override val blockPaths: Seq[BlockPath] = Seq(blockPath)
}
case class SkipSteps(taskId: TaskId, blockPaths: Seq[BlockPath]) extends ModifySteps
case class UnSkipSteps(taskId: TaskId, blockPaths: Seq[BlockPath]) extends ModifySteps

sealed trait ErrorMessage {
  val msg: String
}
case class StepModificationError(msg: String) extends ErrorMessage
case class PathsNotFound(paths:Seq[BlockPath]) extends ErrorMessage {
  val msg: String = s"Paths ${paths.map(_.toBlockId).mkString(", ")} are not found"
}
case class TaskNotFound(taskId: TaskId) extends ErrorMessage {
  override val msg: String = s"Task $taskId was not found. It might have been irrecoverable"
}

sealed trait SuccessMessage
case object PauseStepAdded extends SuccessMessage
case object SkipStepsDone extends SuccessMessage
case object UnSkipStepsDone extends SuccessMessage

// for use with satellite
case class BlockEnvelope(taskId: TaskId, blockPath: BlockPath, message: AnyRef)
case class BlockStateReport(taskId: TaskId, blockId: BlockId, blockState: BlockExecutionState)
case object ReportBlockState
