package com.xebialabs.deployit.engine.tasker

import akka.actor.{Actor, ActorRef, Props}
import com.xebialabs.deployit.engine.api.execution.TaskExecutionState
import com.xebialabs.deployit.engine.tasker.RecoverySupervisorActor.Deleted
import com.xebialabs.deployit.engine.tasker.StateChangeEventListenerActor.{StepStateEvent, TaskStateEvent}
import com.xebialabs.deployit.engine.tasker.messages.TaskStateEventHandled

import scala.language.postfixOps

object TaskRecoveryListener {
  def props(recoverySupervisorActor: ActorRef) = Props(new TaskRecoveryListener(recoverySupervisorActor)).withDispatcher(stateManagementDispatcher)

  val name = "task-recovery-listener"
}

class TaskRecoveryListener(recoverySupervisorActor: ActorRef) extends Actor {

  import context._

  override def preStart(): Unit = {
    system.eventStream.subscribe(self, classOf[TaskStateEvent])
    system.eventStream.subscribe(self, classOf[StepStateEvent])
    super.preStart()
  }

  def receive: Actor.Receive = writerBehaviour

  private def writerBehaviour: Actor.Receive = {
    case TaskStateEvent(_, task, _, state) if !state.isFinal =>
      recoverySupervisorActor ! RecoverySupervisorActor.Write(task)
    case TaskStateEvent(taskId, task, oldState, state) if state.isFinal =>
      recoverySupervisorActor ! RecoverySupervisorActor.Delete(taskId)
      context.become(waitingForDeletionBehaviour(taskId, state, oldState).orElse(writerBehaviour))
    case StepStateEvent(_, _, task, _, _, _, _, _) =>
      recoverySupervisorActor ! RecoverySupervisorActor.Write(task)
  }

  private def waitingForDeletionBehaviour(id: String, state: TaskExecutionState, oldState: TaskExecutionState): Actor.Receive = {
    case Deleted(taskId) if taskId == id =>
      system.eventStream.publish(TaskStateEventHandled(taskId, oldState, state))
      context.become(writerBehaviour)
  }
}