package com.xebialabs.deployit.engine.tasker

import java.io._
import java.nio.file.Files
import java.util
import javax.annotation.{PostConstruct, PreDestroy}

import akka.actor.Actor.Receive
import akka.actor.ActorDSL._
import akka.actor._
import akka.pattern._
import com.github.nscala_time.time.Imports
import com.xebialabs.deployit.engine.api.execution.TaskExecutionState
import com.xebialabs.deployit.engine.api.execution.TaskExecutionState._
import com.xebialabs.deployit.engine.spi.services.RepositoryFactory
import com.xebialabs.deployit.engine.tasker.ArchivedListeningActor.Forward
import com.xebialabs.deployit.engine.tasker.RecoverySupervisorActor.{ReadAll, Tasks}
import com.xebialabs.deployit.engine.tasker.StateChangeEventListenerActor.TaskStateEvent
import com.xebialabs.deployit.engine.tasker.TaskManagingActor.messages._
import com.xebialabs.deployit.engine.tasker.messages._
import com.xebialabs.deployit.plugin.api.services.Repository
import com.xebialabs.xlplatform.settings.CommonSettings
import grizzled.slf4j.Logging

import scala.collection.JavaConverters._
import scala.concurrent.duration._
import scala.concurrent.{Await, Promise}
import scala.language.postfixOps
import scala.util.{Failure, Success, Try}

class TaskExecutionEngine(archive: Archive, repositoryFactory: RepositoryFactory, implicit val system: ActorSystem) extends IEngine with Logging {

  val taskerSettings = CommonSettings(system).tasker

  val archiver = system.actorOf(ArchiveActor.props(archive), ArchiveActor.name)
  val taskTransitioner = system.actorOf(TaskTransitionActor.props(this), TaskTransitionActor.name)
  val recoverySupervisor = system.actorOf(RecoverySupervisorActor.props(taskerSettings.recoveryDir), RecoverySupervisorActor.name)
  val taskRecoveryListener = system.actorOf(TaskRecoveryListener.props(recoverySupervisor), TaskRecoveryListener.name)

  @PreDestroy
  def shutdownTasks() {
    import scala.concurrent.duration._

    TaskRegistryExtension(system).getTasks.filter(t => Set(TaskExecutionState.ABORTING, TaskExecutionState.EXECUTING, TaskExecutionState.FAILING, TaskExecutionState.STOPPING).contains(t.getState)).foreach {
      t =>
        info(s"Stopping task [${t.getId}] due to shutdown")
        lookupTaskActor(t.getId) ! Stop
        val promise = Promise[String]()
        system.actorOf(Props(new Actor {
          override def receive: Receive = {
            case TaskDone(doneTask) if doneTask.getId == t.getId => promise.success(doneTask.getId)
          }
        }))
        Try(Await.result(promise.future, 10 seconds)) match {
          case Failure(exception) => warn(s"Failed to stop task [${t.getId}]", exception)
          case Success(value) => info(s"Successfully stopped task [$value]")
        }
    }

    Await.ready(system.terminate(), taskerSettings.shutdownTimeout)
  }

  @PostConstruct
  def init(): Unit = {
    ActiveTasksQueue(system)
    recoverTasks()
  }

  def recoverTasks() {
    cleanUpTmpFiles()
    try {
      implicit val timeout = taskerSettings.askTimeout
      val tasks = Await.result(recoverySupervisor ? ReadAll, timeout.duration).asInstanceOf[Tasks].tasks
      for (task <- tasks) Try(onRecover(task)).recover {
        case e: Exception => error("Error while recovering tasks.", e)
      }
    } catch {
      case e: Exception => error("Error while recovering.", e)
    }
  }

  private def cleanUpTmpFiles() {
    val tmpFiles = taskerSettings.recoveryDir.listFiles(new FileFilter {
      override def accept(pathname: File): Boolean = pathname.getName.endsWith(".task.tmp")
    })
    Option(tmpFiles).toList.flatten.foreach { file =>
      try {
        warn(s"Deleting corrupted task file: ${file.getAbsolutePath}")
        Files.delete(file.toPath)
      } catch {
        case e: Throwable => error(s"Unable to delete corrupted recovery file: ${file.getAbsolutePath}", e)
      }
    }
  }

  def onRecover(task: Task): Unit = {
    task.context.repository = createRepository(task)
    implicit val timeout = taskerSettings.askTimeout
    Await.ready(createTaskActor(task) ? Recovered(task), timeout.duration)
    if (task.cancelling) {
      cancel(task.getId)
    }
  }

  def archive(taskid: String) {
    archiveWith(taskid, ArchiveTask.apply)
  }

  def archiveWith(taskid: TaskId, msg: (TaskId, ActorRef, ActorRef) => AnyRef): Unit = {
    val p = Promise[TaskId]()
    val listener: ActorRef = system.actorOf(ArchivedListeningActor.props(taskid, p))
    val message: AnyRef = msg(taskid, archiver, listener)

    val taskActor: ActorSelection = lookupTaskActor(taskid)
    listener ! Forward(taskActor, message)
    Await.ready(p.future, Duration.Inf)
    system.stop(listener)
    p.future.value.get match {
      case Failure(ex) => throw ex
      case Success(taskId) => {
        debug(s"Task $taskId successfully archived.")
        TaskRegistryExtension(system).deleteTask(taskid)
      }
    }
  }

  override def skipSteps(taskid: String, stepNrs: util.List[Integer]) {
    skipStepPaths(taskid, stepNrs.asScala.map(int2BlockPath))
  }

  override def unskipSteps(taskid: String, stepNrs: util.List[Integer]) {
    unskipStepPaths(taskid, stepNrs.asScala.map(int2BlockPath))
  }

  override def addPauseStep(taskid: String, position: Integer) {
    addPauseStep(taskid, int2BlockPath(position))
  }

  def addPauseStep(taskid: String, position: BlockPath) {
    manageStep(taskid, AddPauseStep(taskid, position))
  }

  def unskipStepPaths(taskid: String, stepNrs: util.List[BlockPath]) {
    unskipStepPaths(taskid, stepNrs.asScala)
  }
  private[this] def unskipStepPaths(taskid: String, stepNrs: Seq[BlockPath]) {
    manageStep(taskid, UnSkipSteps(taskid, stepNrs))
  }

  def skipStepPaths(taskid: String, stepNrs: util.List[BlockPath]) {
    skipStepPaths(taskid, stepNrs.asScala)
  }

  private[this] def skipStepPaths(taskid: String, stepNrs: Seq[BlockPath]) {
    manageStep(taskid, SkipSteps(taskid, stepNrs))
  }

  private def int2BlockPath(oldPosition: Integer) = BlockPath.Root / 1 / 1 / oldPosition

  private def manageStep(taskid: String, message: ModifySteps) = {
    implicit val timeout = CommonSettings(system).satellite.remoteAskTimeout

    Await.result(lookupTaskActor(taskid) ? message, timeout.duration) match {
      case ex: StepModificationError => throw new TaskerException(ex.msg)
      case PathsNotFound(paths) => throw new TaskerException(s"Cannot find a path to add a pause step at the position $paths")
      case notFound: ActorNotFound => throw new RuntimeException("Probably connection to satellite is lost")
      case _ =>
    }
  }

  def cancel(taskid: String) {
    archiveWith(taskid, Cancel.apply)
  }

  def stop(taskid: String) {
    lookupTaskActor(taskid) ! Stop(taskid)
  }

  def abort(taskid: String) {
    lookupTaskActor(taskid) ! Abort(taskid)
  }

  def execute(taskid: String) {
    val task: Task = retrieve(taskid)
    task.setState(QUEUED)
    lookupTaskActor(taskid) ! Enqueue(taskid)
  }

  def schedule(taskid: String, scheduleAt: com.github.nscala_time.time.Imports.DateTime): Unit = {
    import com.github.nscala_time.time.Imports._
    def p(d: DateTime) = d.toString("yyyy-MM-dd HH:mm:ss Z")
    if (scheduleAt.isBeforeNow) {
      throw new TaskerException(s"Cannot schedule a task for the past, date entered was [${p(scheduleAt)}, now is [${p(DateTime.now)}]")
    }
    val delayMillis: Long = (DateTime.now to scheduleAt).millis
    val tickMillis: Long = taskerSettings.tickDuration.toMillis
    if (delayMillis > Int.MaxValue.toLong * tickMillis) {
      val time: Imports.DateTime = new DateTime(DateTime.now.millis.addToCopy(tickMillis * Int.MaxValue))
      throw new TaskerException(s"Cannot schedule task [$taskid] at [${p(scheduleAt)}], because it is too far into the future. Can only schedule to [${p(time)}]")
    }
    lookupTaskActor(taskid) ! Schedule(taskid, scheduleAt)
  }

  def register(spec: TaskSpecification): String = {
    val task: Task = new Task(spec.getId, spec)
    task.context.repository = createRepository(task)
    implicit val timeout = taskerSettings.askTimeout
    Await.ready(createTaskActor(task) ? Register(task), timeout.duration)
    task.getId
  }

  private def createRepository(task: Task): Repository = {
    repositoryFactory.create(new File(task.getWorkDir.getPath))
  }

  def retrieve(taskid: String): Task = TaskRegistryExtension(system).getTask(taskid).getOrElse(throw new TaskNotFoundException("registry", taskid))

  def getAllIncompleteTasks: util.List[Task] = TaskRegistryExtension(system).getTasks.asJava

  protected[tasker] def lookupTaskActor(s: String): ActorSelection = system.actorSelection(system.child(s))

  protected[tasker] def createTaskActor(task: Task) = system.actorOf(TaskManagingActor.props(task), task.getId)

  def getSystem: ActorSystem = system

}

object ArchivedListeningActor {
  def props(taskId: TaskId, promise: Promise[TaskId]) = Props(classOf[ArchivedListeningActor], taskId, promise)

  case class Forward(actor: ActorSelection, message: AnyRef)

}

class ArchivedListeningActor(taskId: TaskId, promise: Promise[TaskId]) extends Actor {

  import context._

  def receive: Actor.Receive = {
    case Forward(actorSelection, message) =>
      become(identifyActor(message))
      actorSelection ! Identify("")
  }

  def identifyActor(originalMessage: AnyRef): Actor.Receive = {
    case ActorIdentity(_, Some(actorRef)) =>
      watch(actorRef)
      system.eventStream.subscribe(self, classOf[DeadLetter])
      become(await(actorRef))
      system.eventStream.subscribe(self, classOf[TaskStateEvent])
      actorRef ! originalMessage
    case _ =>
      promise.failure(new TaskNotFoundException("akka system", taskId))
  }

  def await(actorRef: ActorRef): Actor.Receive = {
    case Terminated(`actorRef`) | DeadLetter(_, `self`, `actorRef`) =>
      promise.tryFailure(new TaskerException(s"Task $taskId was terminated by a different process"))
    case FailedToArchive(`taskId`, exception) =>
      forget(actorRef)
      promise.tryFailure(new TaskerException(exception, s"Task [$taskId] failed to archive"))
    case akka.actor.Status.Failure(exception) =>
      forget(actorRef)
      promise.tryFailure(new TaskerException(exception, s"Task [$taskId] failed to archive"))
    case Archived(`taskId`) | Cancelled(`taskId`) =>
      forget(actorRef)
      promise.trySuccess(taskId)
    case TaskStateEvent(`taskId`, _, _, TaskExecutionState.FAILED) =>
      forget(actorRef)
      promise.tryFailure(new TaskerException(s"Task [$taskId] failed while cancelling"))
  }

  private def forget(actorRef: ActorRef): Unit = {
    unwatch(actorRef)
    system.eventStream.unsubscribe(self)
  }
}
