package com.xebialabs.deployit.engine.tasker

import java.util.UUID

import akka.actor._
import akka.event.LoggingReceive
import com.xebialabs.deployit.engine.api.execution.{BlockExecutionState, SatelliteConnectionState, StepState}
import com.xebialabs.deployit.engine.tasker.BlockExecutingActor._
import com.xebialabs.deployit.engine.tasker.StateChangeEventListenerActor.{SatelliteStepStateEvent, StepStateEvent}
import com.xebialabs.deployit.engine.tasker.messages._
import com.xebialabs.deployit.engine.tasker.satellite.{ActorLocator, Paths}

class BlockOnSatellite private(val task: Task, block: ExecutableBlock, actorLocator: ActorLocator, notificationActor: ActorRef)
  extends Actor with ModifyStepsSupport with Stash with BecomeWithMdc {

  private val remoteActor = actorLocator.locate(Paths.tasks)(context.system)

  val taskId: TaskId = task.getId

  override def receive: Receive = disconnected(false)

  def disconnected(whileCancelling: Boolean): Receive = {
    case _ if whileCancelling =>
      warn(s"Not reconnecting to satellite ${block.satellite} in cancelling mode. Setting block ${block.id} to DISCONNECTED/DONE")
      updateBlockSatelliteState(SatelliteConnectionState.DISCONNECTED, BlockExecutionState.DONE)
      notificationActor ! BlockDone(taskId, block)
    case _ =>
      tryConnect()
      stash()
  }

  private def tryConnect() = {
    if (block.satelliteState == SatelliteConnectionState.DISCONNECTED) {
      updateBlockSatelliteState(SatelliteConnectionState.RECONNECTING, BlockExecutionState.PENDING)
    }
    val uuid = UUID.randomUUID()
    remoteActor ! Identify(uuid)
    becomeWithMdc(identifyRemoteActor(uuid, sender()))
  }

  private def identifyRemoteActor(uuid: UUID, originalSender: ActorRef): Receive = {
    case ActorIdentity(`uuid`, Some(actorRef)) =>
      debug(s"Remote actor $remoteActor found")
      context.watch(actorRef)
      unstashAll()
      becomeWithMdc(sendModifyStepsAndUpdateBlock(actorRef) orElse sendToSatellite(actorRef) orElse handleBlockDoneOrChanged orElse handleDeathOfRemote() orElse handleTaskNotFound)
    case ActorIdentity(`uuid`, None) =>
      disconnect(s"Could not connect to $remoteActor. Probably, connection to a satellite is broken", Option(originalSender))

    case _ =>
      stash()
  }

  private def handleDeathOfRemote(originalSender: Option[ActorRef] = None): Receive = {
    case Terminated(actorRef) =>
      context.unwatch(actorRef)
      disconnect(s"Remote actor $remoteActor is terminated. Probably, connection to satellite ${block.satellite.get} is broken", originalSender)
  }

  def updateBlockSatelliteState(satelliteState: SatelliteConnectionState, newBlockState: BlockExecutionState): Unit = {
    block.satelliteState(satelliteState)
    val oldState: BlockExecutionState = block.state
    block.newState(newBlockState)
    notificationActor ! BlockStateChanged(taskId, block, oldState, block.state)
  }

  private[this] def handleTaskNotFound(): Receive = {
    case m@TaskNotFound(`taskId`) =>
      error(s"Satellite ${block.satellite.get} reported: ${m.msg}")
      updateBlockSatelliteState(SatelliteConnectionState.UNKNOWN_TASK, BlockExecutionState.FAILED)
      notificationActor ! BlockDone(taskId, block)
      becomeWithMdc(broken)
  }

  private[this] def broken: Receive = {
    case m =>
      error(s"Cannot handle $m, the Satellite ${block.satellite.get} is broken")
      notificationActor ! BlockDone(taskId, block)
  }

  private def disconnect(msg: => String, originalSender: Option[ActorRef] = None) {
    warn(msg)
    becomeWithMdc(disconnected(task.cancelling))
    originalSender.foreach(_.tell(ActorNotFound(remoteActor), self))
    markBlockDisconnected(block)
    notificationActor ! BlockDone(taskId, block)
  }

  private def sendToSatellite(remoteActor: ActorRef): Receive = LoggingReceive {
    case msg@Start(`taskId`) =>
      debug(s"sending $msg to remote: $remoteActor")
      remoteActor ! StartBlock(taskId, block.id)
    case msg@Stop(`taskId`) =>
      debug(s"sending $msg to remote: $remoteActor")
      remoteActor ! StopBlock(taskId, block.id)
    case msg@Abort(`taskId`) =>
      debug(s"sending $msg to remote: $remoteActor")
      remoteActor ! AbortBlock(taskId, block.id)
  }

  private def handleBlockDoneOrChanged: Receive = {
    case BlockDone(`taskId`, updatedBlock: ExecutableBlock) =>
      updateState(block, updatedBlock)
      block.satelliteState(updatedBlock.getSatelliteConnectionState)
      notificationActor ! BlockDone(taskId, block)
    case BlockStateChanged(`taskId`, updatedBlock: ExecutableBlock, oldState, newState) =>
      updateState(block, updatedBlock)
      block.satelliteState(SatelliteConnectionState.CONNECTED)
      notificationActor ! BlockStateChanged(taskId, block, oldState, newState)
  }

  private def updateState(localBlock: ExecutableBlock, remoteBlock: ExecutableBlock) {
    info(s"Changing state of local ${localBlock.id} and remote ${remoteBlock.id} from ${localBlock.state} -> ${remoteBlock.state}")
    (localBlock, remoteBlock) match {
      case (local: StepBlock, remote: StepBlock) =>
        local.newState(remote.state)
        val events = createEvents(local.steps.toList, remote.steps.toList)
        local.steps = remote.steps
        events.foreach { event =>
          context.system.eventStream.publish(SatelliteStepStateEvent(event))
        }

      case (local: CompositeBlock, remote: CompositeBlock) =>
        local.blocks.zip(remote.blocks).foreach {
          case (localSubBlock, remoteSubBlock) =>
            local.newState(remote.state)
            updateState(localSubBlock, remoteSubBlock)
        }
      case _ =>
    }
  }

  private def createEvents(localSteps: List[StepState], satelliteSteps: List[StepState]): List[StepStateEvent] = {
    localSteps.zip(satelliteSteps).flatMap { case (local, satellite) => createEvent(local, satellite) }
  }

  private def createEvent(localStep: StepState, satelliteStep: StepState): Option[StepStateEvent] = {
    if (localStep.getState != satelliteStep.getState) {
      val stepId: String = UUID.randomUUID().toString
      Option(StepStateEvent(taskId, stepId, task, satelliteStep, localStep.getState, satelliteStep.getState, None, None))
    } else {
      None
    }
  }

  private def markBlockDisconnected(localBlock: ExecutableBlock): Unit = localBlock match {
    case cb: CompositeBlock if !cb.state.isFinished =>
      cb.newState(BlockExecutionState.FAILED)
      cb.satelliteState(SatelliteConnectionState.DISCONNECTED)
    case st: StepBlock if !st.state.isFinished =>
      st.newState(BlockExecutionState.FAILED)
      st.satelliteState(SatelliteConnectionState.DISCONNECTED)
    case _ =>
      debug(s"Not marking block $localBlock as disconnected as it is in state ${localBlock.state}")
  }

  private def sendModifyStepsAndUpdateBlock(actorRef: ActorRef): Receive = {
    case msg: ModifySteps if msg.taskId == taskId =>
      debug(s"sending $msg to the remote actor")
      becomeWithMdc(handleDeathOfRemote(Option(sender())) orElse waitForStepModified(taskId, block, context.sender(), msg), discardOld = false)
      actorRef ! BlockEnvelope(taskId = msg.taskId, blockPath = block.id, message = msg)
  }

  private def waitForStepModified(taskId: TaskId, block: ExecutableBlock, originalSender: ActorRef, originalMessage: ModifySteps): Receive = {
    case msg: ErrorMessage =>
      originalSender ! msg
      context.unbecome()
      unstashAll()
    case msg: SuccessMessage =>
      modifySteps(taskId, path => path.relative(block.id).flatMap(block.getBlock), originalSender).apply(originalMessage)
      context.unbecome()
      unstashAll()
    case _ => stash()
  }
}

object BlockOnSatellite {

  def props(task: Task, block: ExecutableBlock, actorLocator: ActorLocator, notificationActor: ActorRef): Props =
    Props(classOf[BlockOnSatellite], task, block, actorLocator, notificationActor).withDispatcher(stateManagementDispatcher)
}
