package com.xebialabs.deployit.engine.tasker

import java.io._
import java.util

import akka.actor.ActorSystem
import com.xebialabs.deployit.engine.api.execution.{StepExecutionState => SES, TaskExecutionState => TES, _}
import com.xebialabs.deployit.engine.tasker.StateChangeEventListenerActor.TaskStateEvent
import com.xebialabs.deployit.engine.tasker.step.PauseStep
import com.xebialabs.deployit.repository.WorkDir
import com.xebialabs.deployit.security.SecretKeyHolder
import grizzled.slf4j.Logger
import org.joda.time.DateTime
import org.joda.time.DateTimeZone.UTC
import org.springframework.security.core.Authentication

import scala.collection.JavaConverters._

object Task extends TaskRecovery {
  private[Task] val logger = Logger.apply(classOf[Task])

  def apply(file: File): Option[Task] = {
    recover(file)(SecretKeyHolder.get())
  }
}


// Wondering whether task shouldn't be an actor itself?
class Task(id: TaskId, spec: TaskSpecification) extends TaskWithSteps with TaskWithBlock with Serializable {

  import com.xebialabs.deployit.engine.tasker.Task.logger

  def this(id: TaskId, workDir: WorkDir, block: PhaseContainer) = this(id, new TaskSpecification("", null, workDir, block))

  private[this] var startDate: DateTime = _
  private[this] var completionDate: DateTime = _
  private[this] var scheduledDate: DateTime = _
  private[this] var owner = spec.getOwner
  private[this] var failureCount = 0
  private[this] var state = TES.UNREGISTERED
  var block: PhaseContainer = spec.getBlock

  var cancelling = false

  private[this] def steps: IndexedSeq[((BlockPath, StepState), Int)] = fetchStepSeq

  val context = new TaskExecutionContext(None, spec)

  if (spec.getInspectionContext != null) {
    spec.getInspectionContext.registerTask(this)
  }

  private[this] def fetchStepSeq: IndexedSeq[((BlockPath, StepState), Int)] = block.getStepsWithPaths().zipWithIndex.toIndexedSeq

  def getSpecification: TaskSpecification = spec

  def getSteps: util.List[StepState] = steps.map(_._1._2).asJava

  def getStep(nr: Int): StepState = steps(stepNrToIndex(nr))._1._2

  def getStep(path: BlockPath): StepState = block.getStep(path.tail)

  def getState: TES = state

  def getId: String = id

  def getDescription: String = spec.getDescription

  def getStartDate: DateTime = startDate

  def getCompletionDate: DateTime = completionDate

  def getNrSteps: Int = steps.size

  def getCurrentStepNr: Int = getCurrentStepNrs.asScala.headOption.map(Integer2int).getOrElse(if (state == TES.EXECUTED) steps.size else 0)

  def getCurrentStepPaths: util.List[BlockPath] = block.getStepsWithPaths().collect({
    case (path, step) if Set(SES.EXECUTING, SES.FAILED, SES.PAUSED).contains(step.getState) => path
  }).asJava

  def getCurrentStepNrs: util.List[Integer] = {
    val mySteps = steps
    def stepByIdxIs(idx: Int, s: SES) = mySteps.find(_._2 == idx).map(_._1._2.getState).contains(s)

    val activeStates = Set(SES.EXECUTING, SES.FAILED, SES.PAUSED)

    // Can't use collect here, as the list changes while we iterate, and the element being considered might change between
    // The pd.isDefinedAt and the pf.apply, resulting in a MatchError
    val list: List[Integer] = steps.view.filter(t => activeStates.contains(t._1._2.getState)).map(t => int2Integer(t._2 + 1)).toList

    val res: util.List[Integer] = (list match {
      case Nil if state == TES.EXECUTED => List(int2Integer(mySteps.size))
      case Nil if Set(TES.STOPPED, TES.ABORTED).contains(state) => mySteps.map(_._2).collect {
        case idx if stepByIdxIs(idx, SES.DONE) && stepByIdxIs(idx + 1, SES.PENDING) => int2Integer(idx + 1)
      }
      case Nil => List(int2Integer(0))
      case x@_ => x
    }).asJava

    res
  }

  def getMetadata: util.Map[String, String] = spec.getMetadata

  def getFailureCount: Int = failureCount

  def getOwner: String = owner.getName

  def getAuthentication: Authentication = owner

  def setOwner(newOwner: Authentication) {
    this.owner = newOwner
  }

  def getWorkDir: WorkDir = spec.getWorkDir

  private[tasker] def setState(state: TES) {
    this.state = state
  }

  private[tasker] def recordFailure() {
    failureCount += 1
  }

  private[tasker] def recordStart() {
    if (startDate == null) {
      startDate = new DateTime(UTC)
    }
  }

  private[tasker] def recordCompletion() {
    completionDate = new DateTime(UTC)
  }

  def getContext: TaskExecutionContext = context

  private[tasker] def getBlock = block

  def getBlock(path: BlockPath): Option[Block] = block.getBlock(path.tail)

  private[tasker] def setTaskStateAndNotify(newState: TES)(implicit system: ActorSystem) {
    val oldState = state
    state = newState
    logger.info(s"Publishing state change $oldState -> $newState")
    system.eventStream.publish(TaskStateEvent(getId, this, oldState, newState))
  }

  def addPause(position: Integer) {
    block.phases match {
      case Seq(Phase(_, _, _, stepBlock: StepBlock)) =>
        if (getCurrentStepNr >= position) throw new IllegalArgumentException(s"Can only add pause steps after the current execution position (currently: [$getCurrentStepNr], requested: [$position])")
        val stepBuffer = stepBlock.steps
        stepBuffer.insert(stepNrToIndex(position), new TaskStep(new PauseStep))

      case _ =>
        throw new TaskerException("Cannot add pause step using old Integer index when the task consists of multiple blocks")
    }
  }

  def recovered() {
    state = state match {
      case TES.EXECUTING | TES.STOPPING | TES.FAILING | TES.ABORTING => TES.FAILED
      case TES.QUEUED => TES.PENDING
      case _ => state
    }

    block.recovered()
  }

  private[tasker] def stepNrToIndex(stepNr: Int): Int = {
    if (stepNr <= 0 || stepNr > steps.size) throw new IllegalArgumentException(s"Not a valid step number [$stepNr]")
    stepNr - 1
  }

  def canBeQueued: Boolean = util.EnumSet.of(TES.PENDING, TES.STOPPED, TES.ABORTED, TES.FAILED).contains(state)

  override def toString: String = s"Task[$id, $state]"

  def getScheduledDate: DateTime = scheduledDate

  def setScheduledDate(scheduledDate: DateTime) {
    this.scheduledDate = scheduledDate
  }

  def getActiveBlocks: util.List[String] = activeBlocks(block).asJava

  val activeBlockStates = Set(BlockExecutionState.EXECUTING, BlockExecutionState.FAILING, BlockExecutionState.STOPPING, BlockExecutionState.ABORTING)

  private[this] def activeBlocks(block: Block): List[String] = block match {
    case PhaseContainer(_, phases) => phases.map(_.block).flatMap(activeBlocks).toList
    case cb: CompositeBlock if activeBlockStates.contains(cb.state) => cb.getId() :: cb.blocks.flatMap(activeBlocks).toList
    case _: CompositeBlock => Nil
    case b: Block if activeBlockStates.contains(b.state) => b.getId() :: Nil
    case _ => Nil
  }

  override def getPackageDependencies: util.List[TaskPackageDependency] = spec.getPackageDependencies
}
