package com.xebialabs.deployit.engine.tasker

import java.util

import com.xebialabs.deployit.engine.api.execution._
import com.xebialabs.deployit.engine.tasker.Phase.{PhaseBuilder, ScheduleInfo}
import com.xebialabs.deployit.engine.tasker.step.PauseStep
import com.xebialabs.deployit.plugin.api.flow.Step
import com.xebialabs.xlplatform.satellite.Satellite
import grizzled.slf4j.Logger

import scala.collection.convert.Wrappers
import scala.collection.mutable.ListBuffer

object Block {
  @transient def logger = Logger(classOf[Block])
}

sealed trait Block extends BlockState {
  def id: BlockPath

  def description: Description

  private[tasker] var state: BlockExecutionState = BlockExecutionState.PENDING
  private[tasker] var satelliteState: SatelliteConnectionState = _


  def getStepList(): java.util.List[StepState]

  def getDescription() = description

  def getId() = id.toBlockId

  def getState() = state

  def getStep(path: BlockPath): StepState

  def getSatelliteConnectionState: SatelliteConnectionState = satelliteState

  def getBlock(path: BlockPath): Option[Block]

  private[tasker] def getStepsWithPaths(): Seq[(BlockPath, StepState)] = this match {
    case x: CompositeBlock => x.blocks.flatMap(b => b.getStepsWithPaths())
    case x: StepBlock => x.steps.zipWithIndex.map({
      case (s, i) => (x.id / (i + 1), s)
    })
    case PhaseContainer(_, phases) => phases.flatMap(_.getStepsWithPaths())
    case Phase(_, _, _, block) => block.getStepsWithPaths()
  }


  def recovered() {
    import com.xebialabs.deployit.engine.api.execution.BlockExecutionState._
    val recoveredState: BlockExecutionState = state match {
      case EXECUTING | STOPPING | FAILING => FAILED
      case ABORTING => ABORTED
      case _ => state
    }
    this.newState(recoveredState)
    this match {
      case x: CompositeBlock => x.blocks.foreach(_.recovered())
      case x: StepBlock => x.steps.foreach(_.asInstanceOf[TaskStep].recovered())
      case PhaseContainer(_, phases) => phases.foreach(_.recovered())
      case Phase(_, _, _, block) => if (block != null) block.recovered()
    }
  }

  // Butt UGLY!
  def newState[A <: Block](state: BlockExecutionState): A = {
    Block.logger.debug(s"Changing state for block $id from ${this.state} -> $state")
    this.state = state
    this.asInstanceOf[A]
  }

  def satelliteState(state: SatelliteConnectionState): Unit = {
    Block.logger.debug(s"Changing state of satellite from ${this.satelliteState} -> ${state}")
    this.satelliteState = state
  }
}

sealed trait ImmutableBlock extends Block {

  import scala.collection.JavaConversions._

  def getStepList(): java.util.List[StepState] = getStepsWithPaths().map(_._2)
}

sealed trait ExecutableBlock extends Block {
  def satellite: Option[Satellite]
}

sealed trait CompositeBlock extends ImmutableBlock with CompositeBlockState with ExecutableBlock {

  import scala.collection.convert.wrapAll._

  def blocks: Seq[ExecutableBlock]

  def determineNewState(block: Block, remainingBlocks: Seq[Block]): BlockExecutionState

  def getBlocks = blocks.toList

  def getBlock(position: BlockPath): Option[Block] = position match {
    case p if p.isEmpty => Some(this)
    case p if p.head <= blocks.size => Option(blocks(p.head - 1)).flatMap(_.getBlock(p.tail))
    case _ => None
  }

  def getStep(position: BlockPath): StepState = position match {
    case p if p.isEmpty => throw new IllegalArgumentException(s"Cannot get step at path [$getId]")
    case p if p.head > blocks.size => throw new IllegalArgumentException(s"Cannot get step [$p] at block [$getId]")
    case p => blocks.get(p.head - 1).getStep(p.tail)
  }
}

case class ParallelBlock(id: BlockPath, description: Description, satellite: Option[Satellite], blocks: Seq[ExecutableBlock]) extends CompositeBlock {

  def determineNewState(block: Block, remainingBlocks: Seq[Block]): BlockExecutionState = {
    import com.xebialabs.deployit.engine.api.execution.BlockExecutionState._
    (state, block.state) match {
      case (PENDING | DONE | FAILED | STOPPED | ABORTED, _) => throw new IllegalStateException(s"Cannot receive state updates when $state")
      case (ABORTING, DONE | FAILED | STOPPED) | (_, ABORTED) => if (remainingBlocks.isEmpty) ABORTED else ABORTING
      case (ABORTING, _) | (_, ABORTING) => ABORTING
      case (FAILING, STOPPED | DONE) | (_, FAILED) => if (remainingBlocks.isEmpty) FAILED else FAILING
      case (FAILING, _) | (_, FAILING) => FAILING
      case (STOPPING, DONE) if remainingBlocks.isEmpty && allDone => DONE
      case (STOPPING, DONE) | (_, STOPPED) => if (remainingBlocks.isEmpty) STOPPED else STOPPING
      case (_, STOPPING) => block.state
      case (EXECUTING, DONE) => if (remainingBlocks.isEmpty) DONE else EXECUTING
      case (_, EXECUTING) => state
    }
  }

  def allDone() = blocks.forall(_.state == BlockExecutionState.DONE)

  def isParallel() = true
}

case class SerialBlock(id: BlockPath, description: Description, satellite: Option[Satellite], blocks: Seq[ExecutableBlock]) extends CompositeBlock {

  def determineNewState(block: Block, remainingBlocks: Seq[Block]): BlockExecutionState = {
    import com.xebialabs.deployit.engine.api.execution.BlockExecutionState._
    (state, block.state) match {
      case (_, DONE) => if (remainingBlocks.isEmpty) DONE else state
      case _ => block.state
    }
  }

  def isParallel() = false

}

case class StepBlock(id: BlockPath, description: Description, satellite: Option[Satellite], var steps: ListBuffer[StepState]) extends Block with StepBlockState with ExecutableBlock {

  import scala.collection.JavaConversions._

  def getStepList(): util.List[StepState] = steps

  def getSteps: util.List[StepState] = steps

  def getCurrentStep: Int = steps.indexWhere(_.getState == StepExecutionState.EXECUTING) + 1

  def addPause(position: BlockPath) {
    position.relative(id) match {
      case Some(BlockPath(List(index))) if index - 1 == steps.size =>
        steps.insert(index - 1, new TaskStep(new PauseStep))
      case Some(BlockPath(List(index))) if index - 1 >= 0 && index - 1 < steps.size
        && !List(StepExecutionState.PENDING, StepExecutionState.SKIP).contains(steps(index - 1).getState) =>
        throw new IllegalArgumentException(s"Add pause step before step [$index] at block [$id]: state must be PENDING or SKIP, was [${steps(index - 1).getState}])")
      case Some(BlockPath(List(index))) if index - 1 >= 0 && index - 1 < steps.size =>
        steps.insert(index - 1, new TaskStep(new PauseStep))
      case _ =>
        throw new IllegalArgumentException(s"Cannot add pause step at $position")
    }
  }

  def skip(stepNrs: Seq[BlockPath]): Unit = {
    stepNrs.map(_.relative(id)).collect {
      case Some(relativePath) =>
        val step: TaskStep = getStep(relativePath).asInstanceOf[TaskStep]
        if (!step.canSkip) throw new IllegalArgumentException(s"Step [$step] cannot be skipped")
        step.setState(StepExecutionState.SKIP)
      case _ =>
        throw new IllegalArgumentException(s"Wrong paths $stepNrs")
    }
  }

  def unskip(stepNrs: Seq[BlockPath]): Unit = {
    stepNrs.map(_.relative(id)).collect {
      case Some(relativePath) =>
        val step: TaskStep = getStep(relativePath).asInstanceOf[TaskStep]
        if (step.hasExecuted) throw new IllegalArgumentException(s"Step [$step] cannot be unskipped")
        step.setState(StepExecutionState.PENDING)
      case _ =>
        throw new IllegalArgumentException(s"Wrong paths $stepNrs")
    }
  }

  def getStep(position: BlockPath): StepState = position match {
    case p if p.isEmpty => throw new IllegalArgumentException(s"No position in step block [$getId] given")
    case p if (p.head - 1) > steps.size => throw new IllegalArgumentException(s"Wrong position [${p.head}] in step block [$getId] given")
    case p => steps.get(p.head - 1)
  }

  override def getBlock(path: BlockPath): Option[Block] = path match {
    case p if p.isEmpty => Some(this)
    case _ => None
  }

}

sealed trait BlockBuilder[BLOCK <: Block] {
  def build(id: BlockPath): BLOCK
}

sealed trait ExecutableBlockBuilder extends BlockBuilder[ExecutableBlock] {

  import com.xebialabs.deployit.engine.tasker.BlockBuilders._

  def build(): PhaseContainer = {
    phases("container", phase("phase-1", "phase", this)).build()
  }

}

trait CompositeBlockBuilder extends ExecutableBlockBuilder

case class PhaseContainerBuilder(description: Description, phases: Seq[PhaseBuilder]) extends BlockBuilder[PhaseContainer] {
  def build() = PhaseContainer(description, phases.zipWithIndex.map {
    case (ph, i) => ph.build(BlockPath.Root / (i + 1))
  })

  def build(ignored: BlockPath) = throw new UnsupportedOperationException("Call build()")
}

case class StepBlockBuilder(description: Description, satellite: Option[Satellite], steps: ListBuffer[StepState]) extends ExecutableBlockBuilder {
  def build(id: BlockPath) = StepBlock(id, description, satellite, steps)
}

case class ParallelBlockBuilder(description: Description, satellite: Option[Satellite], blockBuilders: Seq[ExecutableBlockBuilder]) extends CompositeBlockBuilder {
  def build(id: BlockPath) = ParallelBlock(id, description, satellite, blockBuilders.zipWithIndex.map {
    case (bb, i) => bb.build(id.newSubPath(i + 1))
  })
}

case class SerialBlockBuilder(description: Description, satellite: Option[Satellite], blockBuilders: Seq[ExecutableBlockBuilder]) extends CompositeBlockBuilder {
  def build(id: BlockPath) = SerialBlock(id, description, satellite, blockBuilders.zipWithIndex.map {
    case (bb, i) => bb.build(id.newSubPath(i + 1))
  })
}

object BlockBuilders {

  import scala.collection.convert.wrapAll._

  def phases(description: Description, phaseBuilders: PhaseBuilder*) = PhaseContainerBuilder(description, phaseBuilders)

  def phases(description: Description, phaseBuilders: List[PhaseBuilder]) = PhaseContainerBuilder(description, phaseBuilders)

  def phase(name: String, description: Description, blockBuilder: ExecutableBlockBuilder) = new PhaseBuilder(name, description, blockBuilder)

  def parallel(description: Description, satellite: Option[Satellite], blockBuilders: ExecutableBlockBuilder*) = ParallelBlockBuilder(description, satellite, blockBuilders)

  def parallel(description: Description, satellite: Option[Satellite], blockBuilders: List[ExecutableBlockBuilder]) = ParallelBlockBuilder(description, satellite, blockBuilders)

  def parallel(description: Description, satellite: Satellite, blockBuilders: util.List[ExecutableBlockBuilder]) = ParallelBlockBuilder(description, Option(satellite), blockBuilders)

  def serial(description: Description, satellite: Option[Satellite], blockBuilders: ExecutableBlockBuilder*) = SerialBlockBuilder(description, satellite, blockBuilders)

  def serial(description: Description, satellite: Option[Satellite], blockBuilders: List[ExecutableBlockBuilder]) = SerialBlockBuilder(description, satellite, blockBuilders)

  def serial(description: Description, satellite: Satellite, blockBuilders: util.List[ExecutableBlockBuilder]) = SerialBlockBuilder(description, Option(satellite), blockBuilders)

  def steps(description: Description, satellite: Option[Satellite], steps: List[StepState]) = StepBlockBuilder(description, satellite, ListBuffer.apply(steps: _*))

  def steps(description: Description, satellite: Satellite, steps: util.List[StepState]) = StepBlockBuilder(description, Option(satellite), ListBuffer.apply(steps: _*))

  // auto-convert from Step -> StepState, different name because of generic erasure
  def step(description: Description, steps: List[Step], satellite: Option[Satellite] = None) = StepBlockBuilder(description, satellite, ListBuffer.apply(steps.map(new TaskStep(_)): _*))

  def step(description: Description, satellite: Satellite, steps: util.List[Step]) = StepBlockBuilder(description, Option(satellite), ListBuffer.apply(steps.map(new TaskStep(_)): _*))

}

case class BlockPath(elems: List[Int]) {
  def toBlockId: BlockId = elems.mkString("_")

  def newSubPath(elem: Int): BlockPath = new BlockPath(elems ::: List(elem))

  def /(elem: Int) = newSubPath(elem)

  def tail = new BlockPath(elems.tail)

  def init = new BlockPath(elems.init)

  def head = elems.head

  def isEmpty = elems.isEmpty

  def isLeaf = elems.size == 1

  def take(n: Int) = BlockPath(elems.take(n))

  def depth = elems.size

  def isChild(parentPath: BlockPath) = elems.startsWith(parentPath.elems)

  def isParent(childPath: BlockPath) = childPath.isChild(this)

  def relative(root: BlockPath) = if (this.isChild(root)) Option(BlockPath(this.elems.drop(root.elems.size))) else None

  override def toString: String = s"""BlockPath($toBlockId)"""
}

object BlockPath {
  def apply(pathString: String) = new BlockPath(pathString.split('_').toList.map(_.toInt))

  def apply(elem: Int) = new BlockPath(List(elem))

  object Root extends BlockPath(List(0))

}

object Phase {

  class ScheduleInfo(var scheduledTime: Option[Long]) {

    def scheduled = scheduledTime.isDefined
  }

  trait AlwaysExecuted

  object Schedulable {

    def unapply(phase: Phase): Option[ScheduleInfo] =
      phase.schedule

  }

  object AlwaysExecuted {

    def unapply(phase: Phase): Option[Phase] =
      if (phase.alwaysExecuted) Some(phase) else None
  }

  def unapply(block: Block): Option[(BlockPath, String, Description, ExecutableBlock)] = {
    block match {
      case phase: Phase => Some(phase.id, phase.name, phase.description, phase.getBlock)
      case _ => None
    }
  }

  class PhaseBuilder private(name: String, description: Description, innerBlock: Either[ExecutableBlockBuilder, ExecutableBlock]) extends BlockBuilder[Phase] {

    def this(name: String, description: Description, blockBuilder: ExecutableBlockBuilder) = this(name, description, Left(blockBuilder))

    def this(name: String, description: Description, block: ExecutableBlock) = this(name, description, Right(block))

    private var alwaysExecuted: Boolean = false
    private var schedule = Option.empty[ScheduleInfo]

    override def build(id: BlockPath): Phase = {
      val block: ExecutableBlock = innerBlock match {
        case Right(executableBlock) => executableBlock
        case Left(blockBuilder) => blockBuilder.build(id / 1)
      }

      new Phase(id, name, description, block, schedule, alwaysExecuted)
    }

    def asAlwaysExecuted: PhaseBuilder = {
      this.alwaysExecuted = true
      this
    }

    def asSchedulable: PhaseBuilder = {
      this.schedule = Some(new ScheduleInfo(None))
      this
    }
  }

}

class Phase private(val id: BlockPath,
                    val name: String,
                    val description: Description,
                    val block: ExecutableBlock,
                    private val schedule: Option[ScheduleInfo],
                    private val alwaysExecuted: Boolean) extends Block with PhaseState with Serializable {

  def getBlock: ExecutableBlock = block

  def getBlock(position: BlockPath): Option[Block] = position match {
    case p if p.isEmpty => Some(this)
    case p if p.head == 1 => block.getBlock(p.tail)
    case _ => None
  }

  def getStep(position: BlockPath): StepState = position match {
    case p if p.isEmpty => throw new IllegalArgumentException(s"Cannot get step at path [$getId]")
    case p if p.head == 1 => block.getStep(p.tail)
    case p => throw new IllegalArgumentException(s"Cannot get step [$p] at phase [$getId]")
  }

  import scala.collection.convert.wrapAll._

  def getStepList(): java.util.List[StepState] = getStepsWithPaths().map(_._2)

}

case class PhaseContainer(description: Description, phases: Seq[Phase]) extends Block with PhaseContainerState {

  val id = BlockPath.Root

  import scala.collection.convert.wrapAll._

  override def getBlock(position: BlockPath): Option[Block] = position match {
    case p if p.isEmpty => Some(this)
    case p if p.head <= phases.size => phases.get(p.head - 1).getBlock(p.tail)
    case p => None
  }

  def getStep(position: BlockPath): StepState = position match {
    case p if p.isEmpty => throw new IllegalArgumentException(s"Cannot get step at path [$getId]")
    case p if p.head > phases.size => throw new IllegalArgumentException(s"Cannot get step [$p] at block [$getId]")
    case p => phases.get(p.head - 1).getStep(p.tail)
  }

  def getStepList(): java.util.List[StepState] = getStepsWithPaths().map(_._2)

  def determineNewState(phase: Phase, remainingBlocks: Seq[Phase]): BlockExecutionState = {
    import com.xebialabs.deployit.engine.api.execution.BlockExecutionState._
    (state, phase.state) match {
      case (s, DONE) => if (remainingBlocks.isEmpty) DONE else s
      case _ => phase.state
    }
  }

  def getBlocks = Wrappers.IterableWrapper(phases)

  def getPhases: java.lang.Iterable[Phase] = Wrappers.IterableWrapper(phases)

  def getPhasesState: java.lang.Iterable[PhaseState] = Wrappers.IterableWrapper(phases.map((phase: PhaseState) => phase).toList)

  def this(description: Description, phases: java.util.List[Phase]) = this(description, Wrappers.JListWrapper(phases))


}
