package com.xebialabs.deployit.engine.tasker

import akka.actor.{Actor, ActorRef, Props}
import akka.event.LoggingReceive
import com.xebialabs.deployit.engine.api.execution.BlockExecutionState
import com.xebialabs.deployit.engine.api.execution.BlockExecutionState._
import com.xebialabs.deployit.engine.tasker.BlockExecutingActor.{BlockDone, BlockStateChanged}
import com.xebialabs.deployit.engine.tasker.PhaseContainerExecutingActor.internalMessages.{PhaseCompleted, ProcessRemainingPhases}
import com.xebialabs.deployit.engine.tasker.messages._

object PhaseContainerExecutingActor {
  def props(task: Task, block: PhaseContainer, ctx: TaskExecutionContext, parentRef: ActorRef) = {
    Props(classOf[PhaseContainerExecutingActor], task, block, ctx, parentRef).withDispatcher(stateManagementDispatcher)
  }

  object internalMessages {

    case class PhaseCompleted(phase: Phase)

    case class ProcessRemainingPhases(remainingPhases: Seq[Phase])

  }

}

class PhaseContainerExecutingActor(val task: Task, container: PhaseContainer, ctx: TaskExecutionContext, parentRef: ActorRef) extends BaseExecutionActor with ActorCreationSupport with SplitModifySteps with BecomeWithMdc {
  val taskId = task.getId

  private def logStateChange(newState: String) = debug(s">> {STATE CHANGE}: PhaseContainerExecutingActor => [$newState]. Task: $taskId")

  override def receive: Receive = {
    logStateChange("receive")
    LoggingReceive.withLabel("receive")(sendModifyStepsToBlocks(taskId, lookUpPhaseActor) orElse ReceiveWithMdc(task) {
      case Start(`taskId`) if Set(PENDING, STOPPED, ABORTED, FAILED).contains(container.state) =>
        info(s"[${container.id}] : Processing PhaseContainer of task [${task.getId}]")
        updateStateAndNotify(BlockExecutionState.EXECUTING)
        executePhases(container.phases)

      case Start(`taskId`) =>
        error(s"can't start a container in state ${container.state}")

      case finishUp@FinishUp(`taskId`) =>
        debug(s"[${container.id}] : Received [$finishUp] message, execute finish up phase, for task $taskId")

        val alwaysExecutedPhases = container.phases
          .collect {
            case Phase.AlwaysExecuted(forcedPhase) => forcedPhase
          }.dropWhile(phase => phase.getState() == DONE)
          .toList

        debug(s"Always executed phases for task $taskId are $alwaysExecutedPhases")

        alwaysExecutedPhases match {
          case alwaysExecutedPhase :: tail =>
            task.cancelling = true
            updateStateAndNotify(BlockExecutionState.EXECUTING)
            debug(s"Task ${task.getId} is marked as cancelled")
            executePhases(alwaysExecutedPhases)
          case Nil =>
            updateStateAndNotify(DONE)
            notifyParent()
        }
    })
  }

  private def lookUpPhaseActor(blockPath: BlockPath) = {
    container.phases.find(_.id.isParent(blockPath)).map(p => createOrLookUpPhaseActor(p))
  }

  def updateStateAndNotify(newState: BlockExecutionState) {
    val oldState = container.state
    debug(s"Checking to update state from $oldState -> $newState")
    if (oldState != newState) {
      container.newState(newState)
      val msg = BlockStateChanged(task.getId, container, oldState, newState)
      debug(s"[${container.id}] : Sending BlockStateChanged($oldState->$newState) to ${parentRef.path}")
      parentRef ! msg
    } else {
      debug(s"Not sending BlockStateChanged back to parent because state didnt change. Old: $oldState, new: $newState.")
    }
  }

  def executePhases(phases: Seq[Phase]) {
    val phase = phases.head
    phase.state match {
      case DONE =>
        executePhases(phases.tail)
      case _ =>
        val phaseActor = createOrLookUpPhaseActor(phase)
        info(s"Sending [Start] to phase actor for phase ${phase.id}")
        phaseActor ! Start(taskId)
        becomeWithMdc(waitForPhaseCompletion(phase, phases, phaseActor))
    }
  }

  def createOrLookUpPhaseActor(phase: Phase): ActorRef =
    context.child(phase.getId()) match {
      case Some(ref) =>
        ref
      case None =>
        createChild(PhaseExecutingActor.props(task, phase, ctx), phase.getId())
    }

  def waitForPhaseCompletion(phase: Phase, phases: Seq[Phase], phaseActor: ActorRef): Actor.Receive = {
    logStateChange("waitForPhaseCompletion")
    LoggingReceive.withLabel("waitForPhaseCompletion") {
      case m@BlockDone(`taskId`, `phase`) =>
        debug(s"[${container.id}] : Received $m message for [${phase.id}]")
        parentRef ! PhaseCompleted(phase)
        executeRemainingPhases(phase, phases)

      case ProcessRemainingPhases(remainingPhases) =>
        executePhases(remainingPhases)

      case m@BlockStateChanged(`taskId`, `phase`, oldState, state) =>
        debug(s"[${container.id}] : Received $m message for [${phase.id}]")
        val newState: BlockExecutionState = container.determineNewState(phase, phases)
        debug(s"Determined new container state to be [$newState] for $phase and $phases")
        updateStateAndNotify(newState)

      case stop@Stop(`taskId`) =>
        debug(s"[${container.id}] : Received Stop($taskId) message, stopping sub-block")
        phaseActor ! stop

      case abort@Abort(`taskId`) if phase.getState() == BlockExecutionState.DONE && phases.tail.nonEmpty =>
        debug(s"[${container.id}] : Received [$abort] message, but phase ${phase.id} is already in DONE state so directly updating state to STOPPED")
        updateStateAndNotify(BlockExecutionState.STOPPING)
        updateStateAndNotify(BlockExecutionState.STOPPED)
        notifyParent()

      case abort@Abort(`taskId`) =>
        debug(s"[${container.id}] : Received [$abort] message, aborting sub-block")
        phaseActor ! abort
    }
  }

  def executeRemainingPhases(phase: Phase, phases: Seq[Phase]): Unit = {
    val remainingPhases = phases.tail
    val newState: BlockExecutionState = container.determineNewState(phase, remainingPhases)
    debug(s"Determined new container state to be [$newState] for $phase and $remainingPhases")
    updateStateAndNotify(newState)
    if (newState.isFinished) {
      notifyParent()
    } else {
      self ! ProcessRemainingPhases(remainingPhases)
    }
  }

  def notifyParent() {
    if (container.state != DONE) {
      becomeWithMdc(receive)
    } else {
      doNotAcceptMessages("Container DONE")
    }
    info(s"[${container.id}] : All phases of [${container.id}] completed, notifying parent ($parentRef")
    parentRef ! BlockDone(taskId, container)
  }
}
