package com.xebialabs.deployit.engine.tasker

import java.util.UUID

import akka.actor.{Actor, ActorRef, Props, Stash}
import akka.event.LoggingReceive.withLabel
import akka.pattern.pipe
import com.xebialabs.deployit.engine.api.execution.StepExecutionState._
import com.xebialabs.deployit.engine.api.execution.{StepExecutionState, TaskState}
import com.xebialabs.deployit.engine.tasker.MdcUtils.{mdcWithStep, mdcWithTask, withMdc}
import com.xebialabs.deployit.engine.tasker.StateChangeEventListenerActor.StepStateEvent
import com.xebialabs.deployit.engine.tasker.StepExecutingActor.internalMessages.ExecutionStarted
import com.xebialabs.deployit.engine.tasker.StepExecutingActor.messages._
import com.xebialabs.deployit.engine.tasker.messages.{Abort, StepStateEventHandled}
import com.xebialabs.deployit.plugin.api.flow.StepExitCode._
import com.xebialabs.deployit.plugin.api.flow.{ExecutionContext, Step, StepExitCode}
import com.xebialabs.deployit.repository.WorkDirContext
import com.xebialabs.xlplatform.settings.CommonSettings
import grizzled.slf4j.Logging
import org.springframework.security.core.context.SecurityContextHolder

import scala.concurrent.Future
import scala.concurrent.duration.FiniteDuration

object StepExecutingActor {
  def props(task: TaskState, ctx: TaskExecutionContext) = Props(classOf[StepExecutingActor], task, ctx).withDispatcher(stateManagementDispatcher)

  object messages {

    case class ExecuteStep(taskStep: TaskStep, executionCtx: ExecutionContext)

    case class RetryStep(taskStep: TaskStep, executionContext: ExecutionContext, originalSender: ActorRef)

    case class ExecutedStep(taskStep: TaskStep, executionContext: ExecutionContext, stepExitCode: StepExitCode, originalSender: ActorRef)

    case class FailedStep(taskStep: TaskStep, executionContext: ExecutionContext, t: Throwable, originalSender: ActorRef)

    case object StopRequested
  }

  object internalMessages {

    case class ExecutionStarted(taskStep: TaskStep)

  }

}

class StepExecutingActor(val task: Task, ntec: TaskExecutionContext) extends Actor with Logging with BecomeWithMdc with Stash {
  val settings = CommonSettings(context.system).tasker
  val taskId = task.getId


  @scala.throws[Exception](classOf[Exception])
  override def preStart(): Unit = {
    context.system.eventStream.subscribe(this.self, classOf[StepStateEventHandled])
  }

  def receive = ReceiveWithMdc(task)(receiveExecuteStep())

  def receiveExecuteStep(stopRequested: Boolean = false): Receive = {

    case ExecuteStep(s, _) if s.getState == StepExecutionState.QUEUED || s.getState == StepExecutionState.EXECUTING =>
      throw new IllegalStateException(s"The step [$s] is still either queued or executing, cannot run again.")

    case ExecuteStep(s, _) if s.getState.isFinal =>
      val implementation = s.getImplementation
      debug(s"Will not execute: $implementation with description: ${implementation.getDescription} because it has state: ${s.getState}")
      sender ! s.getState

    case ExecuteStep(s, _) if s.isMarkedForSkip =>
      s.recordStart()
      s.recordCompletion()
      setStepState(s, SKIPPED, sender)

    case message@ExecuteStep(s, ctx) =>
      debug(s"Got ExecuteStep message: $message")
      setStepState(s, StepExecutionState.QUEUED, sender)
      s.clearLog()
      info(s"Queued ${s.getImplementation} for execution")
      executeStep(s, s.getImplementation, ctx)


    case message@RetryStep(s, ctx, originalSender) if stopRequested =>
      debug(s"Got RetryStep: $message, but stop has been requested")
      setStepState(s, PAUSED, originalSender)

    case message@RetryStep(s, ctx, originalSender) =>
      debug(s"Got RetryStep: $message")
      info(s"Continuing ${s.getImplementation}")
      executeStep(s, s.getImplementation, ctx, originalSender, retry = true)

    case StopRequested =>
      debug("Stop has been requested while waiting for an execution message")
      becomeWithMdc(receiveExecuteStep(true))
  }

  private def executeStep(step: TaskStep, implStep: Step, ctx: ExecutionContext, originalSender: ActorRef = sender(), retry: Boolean = false) {
    becomeWithMdc(withLabel("executionStartedOrDoneOrStashAbort")(receiveExecutionStarted orElse receiveExecutionDone() orElse receiveStashAbortStop))
    val f = Future {
      withMdc(mdcWithTask(task) ++ mdcWithStep(implStep)) {
        try {
          if (!retry) {
            info(s"Started ${step.getImplementation}")
            step.recordStart()
          }
          setStepState(step, StepExecutionState.EXECUTING, sender)
          step.registerCurrentThread()
          WorkDirContext.setWorkDir(task.getWorkDir)
          SecurityContextHolder.getContext.setAuthentication(task.getAuthentication)
          self ! ExecutionStarted(step)
          val result: StepExitCode = implStep.execute(ctx)
          step.recordCompletion()
          val executedStep = ExecutedStep(step, ctx, result, originalSender)
          executedStep
        } catch {
          case e: Exception =>
            FailedStep(step, ctx, e, originalSender)
        } finally {
          cleanup(step)
        }
      }
    }(context.system.dispatchers.lookup(stepDispatcher))

    implicit val executionContext = context.system.dispatchers.lookup(stateManagementDispatcher)
    f pipeTo self
  }

  def receiveStopRequested(step: TaskStep): Receive = {
    case StopRequested => becomeWithMdc(withLabel("abortOrExecutionDoneWhenStopRequested")(receiveExecutionDone(stopRequested = true) orElse receiveAbort(step)))
  }

  def receiveExecutionDone(stopRequested: Boolean = false): Receive = {
    case message@ExecutedStep(step, ctx, result, originalSender) =>
      debug(s"Got ExecutedStep: $message")
      result match {
        case SUCCESS =>
          setStepState(step, DONE, originalSender)
        case FAIL =>
          setStepState(step, FAILED, originalSender)
        case PAUSE =>
          setStepState(step, PAUSED, originalSender)
        case RETRY if stopRequested =>
          debug("Retry while stop requested, will pause...")
          setStepState(step, PAUSED, originalSender)
        case RETRY =>
          val delay: FiniteDuration = settings.stepRetryDelay
          ctx.logOutput(s"Retrying in ${delay.toSeconds} seconds...")
          context.system.scheduler.scheduleOnce(delay, self, RetryStep(step, ctx, originalSender))(context.dispatcher)
          becomeWithMdc(withLabel("retryStep")(receiveExecuteStep()))
      }

    case FailedStep(step, ctx, t: Throwable, originalSender) =>
      ctx.logError("Step failed", t)
      step.recordCompletion()
      setStepState(step, FAILED, originalSender)
  }


  def doAbort(step: TaskStep) {
    info(s"[${step.getImplementation.getDescription}] : Received [Abort] message")
    step.interruptRunner()
  }

  def receiveAbort(step: TaskStep): Receive = {
    case Abort(`taskId`) =>
      doAbort(step)
  }

  def receiveExecutionStarted: Receive = {
    case ExecutionStarted(step) =>
      becomeWithMdc(withLabel("abortOrStopOrExecutionDone")(receiveAbort(step) orElse receiveStopRequested(step) orElse receiveExecutionDone()))
      unstashAll()
  }

  def receiveStashAbortStop: Receive = {
    case Abort(`taskId`) =>
      stash()
    case StopRequested =>
      stash()
  }


  private def cleanup(step: TaskStep) = {
    debug(s"Doing cleanup for task $task")
    WorkDirContext.clear()
    SecurityContextHolder.getContext.setAuthentication(null)
    TaskStep.logger.info(step.getState.name)
  }

  private def setStepState(step: TaskStep, state: StepExecutionState, origSender: ActorRef) {
    val oldState: StepExecutionState = step.getState
    step.setState(state)
    val stepId: String = UUID.randomUUID().toString
    val notifyBlock: Boolean = state != EXECUTING
    context.system.eventStream.publish(StepStateEvent(task.getId, stepId, task, step, oldState, state, Some(ntec)))

    if (notifyBlock) {
      becomeWithMdc(withLabel("receiveStepStateEventHandled")(receiveStepStateEventHandled(taskId, stepId, oldState, state, origSender)))
    }
  }

  def receiveStepStateEventHandled(taskId: TaskId, stepId: String, oldState: StepExecutionState, newState: StepExecutionState, blockActor: ActorRef): Receive = {
    case StepStateEventHandled(`taskId`, `stepId`, `oldState`, `newState`) =>
      debug(s"Handled step state change [$oldState->$newState]")
      becomeWithMdc(withLabel("executeStep")(receiveExecuteStep()))
      blockActor ! newState
  }
}
