package com.xebialabs.deployit.engine.tasker

import akka.actor.{Actor, ActorRef, Props}
import com.xebialabs.deployit.engine.api.execution._
import com.xebialabs.deployit.engine.spi.execution.{ExecutionStateListener, NonRemotableExecutionStateListener, StepExecutionStateEvent, TaskExecutionStateEvent}
import com.xebialabs.deployit.engine.tasker.StateChangeEventListenerActor.{SatelliteStepStateEvent, StepStateEvent, TaskStateEvent}
import com.xebialabs.deployit.engine.tasker.messages.{StepStateEventHandled, TaskStateEventHandled}
import grizzled.slf4j.Logging

import scala.util.Try

object StateChangeEventListenerActor {
  val name = "state-listener"
  def props(taskId: TaskId, taskManagingActor: ActorRef) = Props(classOf[StateChangeEventListenerActor], taskId, taskManagingActor).withDispatcher(stateEventListenerDispatcher)
  
    case class TaskStateEvent(taskId: TaskId, t: Task, prev: TaskExecutionState, curr: TaskExecutionState) extends TaskExecutionStateEvent {

      def previousState(): TaskExecutionState = prev

      def currentState(): TaskExecutionState = curr

      def task(): TaskWithSteps = t
    }
    case class StepStateEvent(taskId: TaskId, stepId: StepId, t: Task, stepState: StepState, prev: StepExecutionState, curr: StepExecutionState, ctx: Option[TaskExecutionContext], respondTo: Option[ActorRef]) extends StepExecutionStateEvent {
      def previousState(): StepExecutionState = prev

      def currentState(): StepExecutionState = curr

      def step(): StepState = stepState

      def task(): TaskState = t
    }

    case class SatelliteStepStateEvent(stepStateEvent: StepStateEvent)
}

class StateChangeEventListenerActor(taskId: TaskId, taskManagingActor: ActorRef) extends Actor with Logging {

  def receive: Actor.Receive = ReceiveWithMdc(taskId) {
    case e @ TaskStateEvent(`taskId`, _, _, _) =>
      debug(s"Received [$e]")
      try {
        e.t.getContext.allListeners.foreach { l =>
          debug(s"Notifying listeners [$l] of [$e]")
          Try(l.taskStateChanged(e))
        }
      } catch {
        case e: Exception =>
          error(s"Exception while handling state change: $e")
      }
      taskManagingActor ! TaskStateEventHandled(taskId, e.prev, e.curr)

    case e @ StepStateEvent(`taskId`, stepId, _, _, _, _, _, _) =>
      notifyStateListeners("local", e, stepId)
    case SatelliteStepStateEvent(e @ StepStateEvent(`taskId`, stepId, _, _, _, _, _, _)) =>
      notifyStateListeners("remote", e, stepId, _.isInstanceOf[NonRemotableExecutionStateListener])
  }

  private def notifyStateListeners(where:String, e: StepStateEvent, stepId: StepId, listenerFilter: (ExecutionStateListener) => Boolean = _ => true): Unit = {
    debug(s"Received $where [$e]")
    e.ctx.getOrElse(e.t.context).inheritedListeners.filter(listenerFilter).foreach { l =>
      debug(s"Notifying listeners [$l] of [$e]")
      Try(l.stepStateChanged(e))
    }
    e.respondTo.foreach { _ ! StepStateEventHandled(taskId, stepId, e.prev, e.curr) }
  }
}
