package com.xebialabs.deployit.engine.tasker

import java.io.{PrintWriter, Serializable, StringWriter}
import java.util

import com.xebialabs.deployit.engine.spi.execution.ExecutionStateListener
import com.xebialabs.deployit.plugin.api.execution.ExecutionContextListener
import com.xebialabs.deployit.plugin.api.flow.{ExecutionContext, ITask}
import com.xebialabs.deployit.plugin.api.inspection.InspectionContext
import com.xebialabs.deployit.plugin.api.services.Repository
import grizzled.slf4j.Logger

import scala.collection.convert.wrapAll._

class TaskExecutionContext(parent: Option[TaskExecutionContext], spec: TaskSpecification) extends Serializable {
  private[this] val attributes: Cache[String, AnyRef] = Cache()

  attributes += (TaskExecutionContext.CACHE_KEY -> this.spec)

  @transient var repository: Repository = null

  if (parent.isEmpty && spec != null) {
    spec.getListeners.foreach(l => attributes.put(l.getClass.getName, l))
  }

  private def getParent: Option[TaskExecutionContext] = parent

  val childContexts: Cache[BlockPath, TaskExecutionContext] = Cache()

  def contextFor(block: Block): TaskExecutionContext = contextFor(block.id)

  def contextFor(path: BlockPath): TaskExecutionContext = childContexts.getOrElse(path, {
    val c = new TaskExecutionContext(Some(this), spec)
    childContexts.put(path, c)
    c
  })

  private[TaskExecutionContext] def listeners = attributes.collect {
    case (_, l: ExecutionStateListener) => l
  }.toSet

  private[tasker] def allListeners: Set[ExecutionStateListener] = {
    def getRootContext(tec: TaskExecutionContext): TaskExecutionContext = tec.getParent match {
      case Some(pp) => getRootContext(pp)
      case None => tec
    }

    def recursiveCollectListeners(t: TaskExecutionContext): Set[ExecutionStateListener] = t.listeners ++ t.childContexts.values.flatMap(recursiveCollectListeners)
    recursiveCollectListeners(getRootContext(this))
  }

  private[tasker] def inheritedListeners: Set[ExecutionStateListener] = parent match {
    case None => listeners
    case Some(p) => listeners ++ p.inheritedListeners
  }

  def clearChildContexts() = childContexts.clear()

  def getAttribute(name: String): AnyRef = attributes.getOrElse(name, parent.map(_.getAttribute(name)).orNull) match {
    case l: OldExecutionContextListenerCleanupTrigger => l.getWrappedListener
    case l@_ => l
  }

  def setAttribute(name: String, v: AnyRef) = v match {
    case ecl: ExecutionContextListener => attributes.put(name, new OldExecutionContextListenerCleanupTrigger(name, ecl))
    case x@_ => attributes.put(name, x)
  }

  private[tasker] def getAttributes = attributes

  def unsetAttribute(name: String) = attributes.remove(name)

  def getRepository = repository

  def stepContext(step: TaskStep, task: Task) = new ExecutionContext {
    val ERROR_PREFIX = "[ERROR]: "

    val stepLogger = Logger.apply(step.getImplementation.getClass)

    override def setAttribute(name: String, value: scala.AnyRef) {
      TaskExecutionContext.this.setAttribute(name, value)
    }

    override def logOutput(output: String) {
      logOutputRaw(output + "\n")
    }

    override def logOutputRaw(output: String): Unit = {
      stepLogger.info(output.stripSuffix("\n"))
      step.logBuilder.append(output)
      step.touch()
    }

    override def logError(error: String): Unit = {
      logErrorRaw(error + "\n")
    }

    override def logErrorRaw(error: String): Unit = {
      logErrorRaw(error, None)
    }

    def logError(error: String, t: Throwable): Unit = {
      logErrorRaw(error + "\n", Option(t))
    }

    private def logErrorRaw(error: String, t: Option[Throwable]): Unit = {
      if (t.isDefined) {
        stepLogger.error(error.stripSuffix("\n"), t.get)
      } else {
        stepLogger.error(error.stripSuffix("\n"))
      }
      appendErrorToLogRaw(error)
      t.foreach(logException)
      step.touch()
    }

    private def logException(t: Throwable): Unit = {
      val stringWriter: StringWriter = new StringWriter
      t.printStackTrace(new PrintWriter(stringWriter))
      appendErrorToLogRaw(stringWriter.toString)
    }

    private def appendErrorToLogRaw(error: String) {
      step.logBuilder.append(ERROR_PREFIX).append(error)
    }

    override def getAttribute(name: String): AnyRef = attributes.getOrElse(name, null) match {
      case l: OldExecutionContextListenerCleanupTrigger => l.getWrappedListener
      case l@_ => l
    }

    override def getInspectionContext: InspectionContext = spec.getInspectionContext

    override def getRepository: Repository = repository

    override def getTask: ITask = new ITask {

      override def getMetadata: util.Map[String, String] = task.getMetadata

      override def getUsername: String = task.getOwner

      override def getId: String = task.getId
    }

  }

}

object TaskExecutionContext {
  val CACHE_KEY = "taskExecutionContext"

}