package com.xebialabs.deployit.engine.tasker

import akka.actor._
import com.xebialabs.deployit.engine.tasker.messages._
import grizzled.slf4j.Logging

import scala.util.{Failure, Success, Try}

trait ModifyStepsSupport extends Logging with Actor {

  type BlockFinder = BlockPath => Option[Block]

  def forwardStepModification(taskId: TaskId, actorLookUp: () => ActorRef): Actor.Receive = {
    case msg: ModifySteps if msg.taskId == taskId =>
      actorLookUp() forward msg
  }

  def modifySteps(taskId: TaskId, blockFinder: BlockPath => Option[Block], sender: () => ActorRef = context.sender): Actor.Receive = {
    case msg@AddPauseStep(`taskId`, blockPath) =>
      debug(s"Adding pause step message for $msg")
      addPauseStep(blockFinder, blockPath, sender())
    case msg@SkipSteps(`taskId`, blockPaths) =>
      debug(s"Adding skip step message for $msg")
      skipSteps(blockFinder, blockPaths, sender())
    case msg@UnSkipSteps(`taskId`, blockPaths) =>
      debug(s"Adding unskip step message for $msg")
      unSkipSteps(blockFinder, blockPaths, sender())
  }

  private def addPauseStep(blockFinder: BlockFinder, blockPath: BlockPath, sender: ActorRef) = {
    findBlockAndTry(PauseStepAdded, Seq(blockPath), sender)(blockFinder, _.addPause(blockPath))
  }

  private def skipSteps(blockFinder: BlockFinder, blockPaths: Seq[BlockPath], sender: ActorRef) = {
    findBlockAndTry(SkipStepsDone, blockPaths, sender)(blockFinder, _.skip(blockPaths))
  }

  private def unSkipSteps(blockFinder: BlockFinder, blockPaths: Seq[BlockPath], sender: ActorRef) = {
    findBlockAndTry(UnSkipStepsDone, blockPaths, sender)(blockFinder, _.unskip(blockPaths))
  }

  private def findBlockAndTry[T](successMessage: SuccessMessage, blockPaths: Seq[BlockPath], sender: ActorRef)
                                (blockFinder: BlockPath => Option[Block], tryOperation: StepBlock => T): Unit = {
    def doTry(block: StepBlock): Unit = {
      Try(tryOperation(block)) match {
        case Success(_) => sender ! successMessage
        case Failure(ex) => sender ! StepModificationError(ex.getMessage)
      }
    }

    def doForSingleBlockPath(blockPath: BlockPath): Unit = {
      blockFinder(blockPath) match {
        case Some(sb: StepBlock) => doTry(sb)
        case _ => sender ! PathsNotFound(Seq(blockPath))
      }
    }

    blockPaths.map(_.init).toSet[BlockPath].headOption.collect {
      case path => doForSingleBlockPath(path)
    }.getOrElse(sender ! PathsNotFound(blockPaths))
  }
}
