package com.xebialabs.deployit.engine.tasker

import akka.actor.Status.Failure
import akka.actor._
import akka.event.LoggingReceive
import com.github.nscala_time.time.Imports._
import com.xebialabs.deployit.engine.api.execution.TaskExecutionState._
import com.xebialabs.deployit.engine.api.execution.{BlockExecutionState => BlockIs, TaskExecutionState}
import com.xebialabs.deployit.engine.tasker.ArchiveActor.messages.SendToArchive
import com.xebialabs.deployit.engine.tasker.BlockExecutingActor.{BlockDone, BlockStateChanged}
import com.xebialabs.deployit.engine.tasker.StateChangeEventListenerActor.{StepStateEvent, TaskStateEvent}
import com.xebialabs.deployit.engine.tasker.TaskManagingActor.messages.{ArchiveTask, Cancel, Recovered, Register, Schedule, ScheduledStart}
import com.xebialabs.deployit.engine.tasker.messages._
import org.joda.time.DateTime

import scala.concurrent.duration._

object TaskManagingActor {
  def props(task: Task) = Props(classOf[TaskManagingActor], task)

  def getMessages = messages

  object messages {

    case class Register(task: Task)

    case class ScheduledStart(taskId: TaskId)

    case class Schedule(taskId: TaskId, scheduleAt: DateTime)

    case class ArchiveTask(taskId: TaskId, archiveActor: ActorRef, notificationActor: ActorRef)

    case class Cancel(taskId: TaskId, archiveActor: ActorRef, notificationActor: ActorRef)

    case class Recovered(task: Task)

  }

}

class TaskManagingActor(val task: Task) extends BaseExecutionActor with Stash with ModifyStepsSupport with BecomeWithMdc {

  import context._

  val taskId = task.getId

  val terminationStates = Set(FAILED, ABORTED, EXECUTED, STOPPED)

  private val blockFinder: BlockFinder = task.getBlock

  private def logStateChange(newState: String) = debug(s">> {STATE CHANGE}: TaskManagingActor => [$newState]. Task: $taskId.")

  def receive: Actor.Receive = {
    logStateChange("receive")
    ReceiveWithMdc()(LoggingReceive.withLabel("receive") {
      case Register(`task`) =>
        info(s"Received [Register] message for task [$taskId]")
        registerStateListeners(task)

        updateStateAndNotify(PENDING)
        becomeWithMdc(pending)

        TaskRegistryExtension(system).register(task)
        sender() ! Registered(taskId)
      case Recovered(`task`) =>
        info(s"Received [Recovered] message for task [$taskId]")
        registerStateListeners(task)
        task.getState match {
          case PENDING | QUEUED =>
            becomeWithMdc(pending)
          case SCHEDULED if task.getScheduledDate.isAfterNow =>
            doSchedule(task.getScheduledDate, createOrLookupChildForTaskBlock())
          case SCHEDULED =>
            doEnqueue(createOrLookupChildForTaskBlock())
          case EXECUTED =>
            becomeWithMdc(canBeArchived)
          case STOPPED | FAILED | ABORTED =>
            becomeWithMdc(readyForRestart(createOrLookupChildForTaskBlock()))
          case _ =>
            sender() ! Failure(new IllegalStateException(s"Cannot recover a task which is in state [${task.getState}]."))
        }
        TaskRegistryExtension(system).register(task)
        sender() ! Registered(taskId)
    })
  }

  def pending: Actor.Receive = {
    logStateChange("pending")
    LoggingReceive.withLabel("pending")(modifySteps orElse ReceiveOrFail {
      case Enqueue(`taskId`) =>
        doEnqueue(createOrLookupChildForTaskBlock())
      case Schedule(`taskId`, scheduleAt) =>
        doSchedule(scheduleAt, createOrLookupChildForTaskBlock())
      case Cancel(`taskId`, archiveActor, notificationActor) =>
        doCancelWhenPending(Some(notificationActor))
    })
  }

  def queued(blockActor: ActorRef): Actor.Receive = {
    logStateChange("queued")
    LoggingReceive.withLabel("queued")(ReceiveOrFail {
      case Start(`taskId`) =>
        doStart(blockActor)
    })
  }

  def scheduled(scheduleHandle: Cancellable, blockActor: ActorRef): Actor.Receive = {
    logStateChange("scheduled")
    LoggingReceive.withLabel("scheduled") {
      case Enqueue(`taskId`) =>
        scheduleHandle.cancel()
        doEnqueue(blockActor)
      case Schedule(`taskId`, scheduleAt) =>
        scheduleHandle.cancel()
        doSchedule(scheduleAt, blockActor)
      case ScheduledStart(`taskId`) =>
        doEnqueue(blockActor)
      case Cancel(`taskId`, _, notificationActor) if task.getStartDate == null =>
        scheduleHandle.cancel()
        doCancelWhenPending(Some(notificationActor))
      case Cancel(`taskId`, archiveActor, notificationActor) if task.getStartDate != null =>
        scheduleHandle.cancel()
        doCancel(archiveActor, Some(notificationActor))
      case event: TaskStateEventHandled =>
      case msg:AnyRef =>
        throw new IllegalStateException(msg.toString)
    }
  }

  def executing(blockActor: ActorRef): Actor.Receive = {
    logStateChange("executing")
    LoggingReceive.withLabel("executing")(ReceiveOrFail {
      case m@BlockStateChanged(`taskId`, block, oldState, state) =>
        debug(s"executing(), Received $m")
        state match {
          case BlockIs.PENDING | BlockIs.EXECUTING =>
          case BlockIs.DONE =>
            updateStateAndNotify(EXECUTED)
            becomeWithMdc(executed)
          case BlockIs.FAILING =>
            updateStateAndNotify(FAILING)
            becomeWithMdc(failing(blockActor, readyForRestart(blockActor)))
          case BlockIs.FAILED =>
            updateStateAndNotify(FAILED)
            becomeWithMdc(failed(blockActor, forwardStepModification orElse readyForRestart(blockActor)))
          case BlockIs.STOPPING =>
            updateStateAndNotify(STOPPING)
            becomeWithMdc(stopping(blockActor, readyForRestart(blockActor)))
          case BlockIs.STOPPED =>
            updateStateAndNotify(STOPPED)
            becomeWithMdc(stopped(blockActor, forwardStepModification orElse readyForRestart(blockActor)))
          case BlockIs.ABORTING =>
            updateStateAndNotify(ABORTING)
            becomeWithMdc(aborting(blockActor, readyForRestart(blockActor)))
          case BlockIs.ABORTED =>
            updateStateAndNotify(ABORTED)
            becomeWithMdc(aborted(blockActor, forwardStepModification orElse readyForRestart(blockActor)))
        }
      case s@Stop(`taskId`) =>
        debug(s"Received [$s], now stopping execution")
        blockActor ! s
      case m@Abort(`taskId`) =>
        debug(s"Received [$m], now aborting execution")
        blockActor ! m
    })
  }

  def executed: Actor.Receive = {
    logStateChange("executed")
    LoggingReceive.withLabel("executed")(ReceiveOrFail {
      case m@BlockDone(`taskId`, block) =>
        debug(s"executed(), Received $m")
        task.recordCompletion()
        notifyTaskDone()
        becomeWithMdc(canBeArchived)
        unstashAll()
      case ArchiveTask(`taskId`, _, _) => stash()
    })
  }

  def canBeArchived: Actor.Receive = {
    logStateChange("canBeArchived")
    LoggingReceive.withLabel("canBeArchived")(ReceiveOrFail {
      case ArchiveTask(`taskId`, archiveActor: ActorRef, notificationActor: ActorRef) =>
        info(s"Received [ArchiveTask] message for task [$taskId]")
        updateStateAndNotify(DONE)
        becomeWithMdc(waitForStateHandledThenArchive(DONE, archiveActor, Some(notificationActor)))
    })
  }

  def archiving(notificationActor: Option[ActorRef]): Actor.Receive = {
    logStateChange("archiving")
    LoggingReceive.withLabel("archiving") {
      case Archived(`taskId`) =>
        notifyTaskDone()
        TaskRegistryExtension(system).deleteTask(taskId)
        task.getState match {
          case DONE => notificationActor.foreach{ _  ! Archived(taskId) }
          case CANCELLED => notificationActor.foreach{ _ ! Cancelled(taskId) }
          case _@state => warn(s"Unexpected state: $state for task: $taskId after task archive operation.")
        }
        harakiri(s"Done with task [$taskId]")

      case fta@FailedToArchive(`taskId`, exception) =>
        info(s"Task [$taskId] failed to archive, going back to previous state.")
        task.getState match {
          case DONE =>
            updateStateAndNotify(EXECUTED)
            becomeWithMdc(executed)
          case CANCELLED =>
            updateStateAndNotify(FAILED)
            val blockActor = createOrLookupChildForTaskBlock()
            becomeWithMdc(failed(blockActor, readyForRestart(blockActor)))
          case _@state => warn(s"Unexpected state: $state for task: $taskId when failed to archive.")
        }
        notifyTaskDone()
        notificationActor.foreach { _ ! fta }
      case _@message => warn(s"Unexpected message during task: $taskId is being archived.")
    }
  }

  def failing(blockActor: ActorRef, restartState: => Receive): Actor.Receive = {
    logStateChange("failing")
    LoggingReceive.withLabel("failing")(ReceiveOrFail {
      case m@BlockStateChanged(`taskId`, block, oldState, state) =>
        debug(s"failing(), received $m")
        state match {
          case BlockIs.PENDING | BlockIs.EXECUTING | BlockIs.DONE | BlockIs.FAILING | BlockIs.STOPPING | BlockIs.STOPPED =>
          case BlockIs.FAILED =>
            updateStateAndNotify(FAILED)
            becomeWithMdc(failed(blockActor, forwardStepModification orElse restartState))
          case BlockIs.ABORTING =>
            updateStateAndNotify(ABORTING)
            becomeWithMdc(aborting(blockActor, restartState))
          case BlockIs.ABORTED =>
            updateStateAndNotify(ABORTED)
            becomeWithMdc(aborted(blockActor, forwardStepModification orElse restartState))
        }
      case s@Stop(`taskId`) =>
        debug(s"Received [$s], now stopping execution")
        blockActor ! s
      case m@Abort(`taskId`) =>
        debug(s"Received [$m], now aborting execution")
        blockActor ! m
    })
  }

  def failed(blockActor: ActorRef, restartState: => Receive): Actor.Receive = {
    logStateChange("failed")
    LoggingReceive.withLabel("failed")(ReceiveOrFail {
      case m@BlockDone(`taskId`, block) =>
        debug(s"failed(), received $m")
        task.recordFailure()
        task.recordCompletion()
        notifyTaskDone()
        becomeWithMdc(restartState)
        unstashAll()
      case Enqueue(`taskId`) | Schedule(`taskId`, _) | Cancel(`taskId`, _, _) => stash()
    })
  }

  def readyForRestart(blockActor: ActorRef): Actor.Receive = {
    logStateChange("readyForRestart")
    LoggingReceive.withLabel("readyForRestart")(modifySteps orElse ReceiveOrFail {
      case Enqueue(`taskId`) =>
        doEnqueue(blockActor)

      case Schedule(`taskId`, scheduleAt) =>
        doSchedule(scheduleAt, blockActor)

      case Cancel(`taskId`, archiveActor, notificationActor) =>
        doCancel(archiveActor, Some(notificationActor))
    })
  }

  def cancelling(blockActor: ActorRef, waitForStateHandle: => Receive, readyState: => Receive): Actor.Receive = {
    logStateChange("cancelling")
    LoggingReceive.withLabel("cancelling")(ReceiveOrFail {
      case m@BlockStateChanged(`taskId`, block, oldState, state) =>
        debug(s"cancelling(), received $m")
        state match {
          case BlockIs.PENDING | BlockIs.EXECUTING =>
          case BlockIs.DONE =>
            task.recordCompletion()
            updateStateAndNotify(CANCELLED)
            notifyTaskDone()
            TaskRegistryExtension(system).deleteTask(taskId)
            becomeWithMdc(waitForStateHandle)
          case BlockIs.FAILING =>
            updateStateAndNotify(FAILING)
            becomeWithMdc(failing(blockActor, readyState))
          case BlockIs.FAILED =>
            task.recordCompletion()
            updateStateAndNotify(FAILED)
            becomeWithMdc(failed(blockActor, forwardStepModification orElse readyState))
          case BlockIs.STOPPING =>
            updateStateAndNotify(STOPPING)
            becomeWithMdc(stopping(blockActor, readyState))
          case BlockIs.STOPPED =>
            task.recordCompletion()
            updateStateAndNotify(STOPPED)
            becomeWithMdc(stopped(blockActor, forwardStepModification orElse readyState))
          case BlockIs.ABORTING =>
            updateStateAndNotify(ABORTING)
            becomeWithMdc(aborting(blockActor, readyState))
          case BlockIs.ABORTED =>
            task.recordCompletion()
            updateStateAndNotify(ABORTED)
            becomeWithMdc(aborted(blockActor, forwardStepModification orElse readyState))

        }
      case s@Stop(`taskId`) =>
        debug(s"Received [$s], now stopping execution")
        blockActor ! s
      case m@Abort(`taskId`) =>
        debug(s"Received [$m], now aborting execution")
        blockActor ! m
    })
  }

  def readyForCancelling(archiveActor: ActorRef): Receive = {
    logStateChange("readyForCancelling")
    LoggingReceive.withLabel("readyForCancelling") {
      case Enqueue(`taskId`) =>
        doCancel(archiveActor, None)
      case Cancel(`taskId`, aA, nA) =>
        doCancel(aA, Some(nA))
    }
  }

  def waitForStateHandledThenArchive(state: TaskExecutionState, archiveActor: ActorRef, notificationActor: Option[ActorRef]): Actor.Receive = {
    logStateChange("waitForStateHandledThenArchive")
    LoggingReceive.withLabel("waitForStateHandledThenArchive") {
      case TaskStateEventHandled(`taskId`, _, `state`) if !task.getSpecification.isArchiveable =>
        becomeWithMdc(archiving(notificationActor))
        self ! Archived(task.getId)
      case TaskStateEventHandled(`taskId`, _, `state`) =>
        becomeWithMdc(archiving(notificationActor))
        archiveActor ! SendToArchive(task, self)
      case _@message =>
        debug(s"warn message: {$message} when waiting for state handled before archive.")
    }
  }

  def stopping(blockActor: ActorRef, restartState: => Receive): Actor.Receive = {
    logStateChange("stopping")
    LoggingReceive.withLabel("stopping")(ReceiveOrFail {
      case BlockStateChanged(`taskId`, block, oldState, state) =>
        state match {
          case BlockIs.PENDING | BlockIs.EXECUTING | BlockIs.STOPPING =>
          case BlockIs.DONE =>
            updateStateAndNotify(EXECUTED)
            becomeWithMdc(executed)
          case BlockIs.FAILING =>
            updateStateAndNotify(FAILING)
            becomeWithMdc(failing(blockActor, restartState))
          case BlockIs.FAILED =>
            updateStateAndNotify(FAILED)
            becomeWithMdc(failed(blockActor, forwardStepModification orElse restartState))
          case BlockIs.STOPPED =>
            updateStateAndNotify(STOPPED)
            becomeWithMdc(stopped(blockActor, forwardStepModification orElse restartState))
          case BlockIs.ABORTING =>
            updateStateAndNotify(ABORTING)
            becomeWithMdc(aborting(blockActor, restartState))
          case BlockIs.ABORTED =>
            updateStateAndNotify(ABORTED)
            becomeWithMdc(aborted(blockActor, forwardStepModification orElse restartState))
        }
      case s@Stop(`taskId`) =>
        debug(s"Received [$s], now stopping execution")
        blockActor ! s
      case m@Abort(`taskId`) =>
        debug(s"Received [$m], now aborting execution")
        blockActor ! m
    })
  }

  def stopped(blockActor: ActorRef, restartState: => Receive): Actor.Receive = {
    logStateChange("stopped")
    LoggingReceive.withLabel("stopped")(ReceiveOrFail {
      case m@BlockDone(`taskId`, block) =>
        debug(s"stopped(), received $m")
        task.recordCompletion()
        notifyTaskDone()
        becomeWithMdc(restartState)
        unstashAll()
      case Enqueue(`taskId`) | Schedule(`taskId`, _) | Cancel(`taskId`, _, _) => stash()
    })
  }

  def aborting(blockActor: ActorRef, restartState: => Receive): Actor.Receive = {
    logStateChange("aborting")
    LoggingReceive.withLabel("aborting")(ReceiveOrFail {
      case BlockStateChanged(`taskId`, block, oldState, state) =>
        state match {
          case BlockIs.PENDING | BlockIs.EXECUTING | BlockIs.DONE | BlockIs.FAILING | BlockIs.STOPPING | BlockIs.STOPPED | BlockIs.FAILED | BlockIs.ABORTING =>
          case BlockIs.ABORTED =>
            updateStateAndNotify(ABORTED)
            becomeWithMdc(aborted(blockActor, forwardStepModification orElse restartState))
        }
    })
  }

  def aborted(blockActor: ActorRef, restartState: => Receive): Actor.Receive = {
    logStateChange("aborted")
    LoggingReceive.withLabel("aborted")(ReceiveOrFail {
      case m@BlockDone(`taskId`, block) =>
        debug(s"aborted(), received $m")
        task.recordFailure()
        task.recordCompletion()
        notifyTaskDone()
        becomeWithMdc(restartState)
        unstashAll()
      case Enqueue(`taskId`) | Schedule(`taskId`, _) | Cancel(`taskId`, _, _) => stash()
    })
  }

  def registerStateListeners(task: Task) {
    def registerStateListener(child: ActorRef) {
      context.system.eventStream.subscribe(child, classOf[TaskStateEvent])
      context.system.eventStream.subscribe(child, classOf[StepStateEvent])
    }

    val child: ActorRef = createChild(StateChangeEventListenerActor.props(taskId, self), "state-listener")
    registerStateListener(child)
  }

  def calculateDelay(time: DateTime): FiniteDuration = FiniteDuration((DateTime.now to time).millis, MILLISECONDS)

  def doEnqueue(blockActor: ActorRef) {
    info(s"Received [Enqueue] message for task [$taskId]")
    updateStateAndNotify(QUEUED)
    becomeWithMdc(queued(blockActor))
    self ! Start(taskId)
  }

  def doStart(blockActor: ActorRef) {
    info(s"Received [Start] message for task [$taskId]")
    task.recordStart()
    updateStateAndNotify(EXECUTING)
    becomeWithMdc(executing(blockActor))
    info(s"sending ${Start(taskId)} to $blockActor")
    blockActor ! Start(taskId)
  }

  def doSchedule(scheduleAt: DateTime, blockActor: ActorRef) {
    info(s"Received [Schedule] message for task [$taskId]")
    task.setScheduledDate(scheduleAt)
    updateStateAndNotify(SCHEDULED)

    val delay: FiniteDuration = calculateDelay(scheduleAt)
    info(s"Going to schedule task [$taskId] at [${scheduleAt.toString("yyyy-MM-dd hh:mm:ss Z")}] which is [$delay] from now")

    if (delay.toMinutes < 0) {
      doEnqueue(blockActor)
    } else {
      val scheduleHandle: Cancellable = context.system.scheduler.scheduleOnce(delay, self, Enqueue(taskId))
      becomeWithMdc(scheduled(scheduleHandle, blockActor))
    }
  }

  def doCancelWhenPending(notificationActor: Option[ActorRef]) {
    info(s"Received [Cancel] message for task [$taskId]")
    updateStateAndNotify(CANCELLED)
    notifyTaskDone()
    TaskRegistryExtension(system).deleteTask(taskId)
    becomeWithMdc({
      case TaskStateEventHandled(`taskId`, _, CANCELLED) =>
        notificationActor.foreach {
          _ ! Cancelled(taskId)
        }
      case _@message =>
        warn(s"warn message: {$message} while waiting for cancelled state event")
    })
  }

  def waitForCancelWithoutArchive(notificationActor: ActorRef): Receive = {
    logStateChange("waitForCancelWithoutArchive")
    LoggingReceive.withLabel("waitForCancelWithoutArchive") {
      case TaskStateEventHandled(`taskId`, _, CANCELLED) =>
        notificationActor ! Cancelled(taskId)
      case _@message =>
        warn(s"Unexpected message: {$message} while waiting for cancel without archiving")
    }
  }

  def doCancel(archiveActor: ActorRef, notificationActor: Option[ActorRef]) {
    info(s"Received [Cancel] message for task [$taskId]")

    val blockActor = createOrLookupChildForTaskBlock()
    updateStateAndNotify(CANCELLING)

    val waitForStateHandled = waitForStateHandledThenArchive(CANCELLED, archiveActor, notificationActor)
    val readyState = readyForCancelling(archiveActor)

    becomeWithMdc(cancelling(blockActor, waitForStateHandled, readyState))

    blockActor ! FinishUp(taskId)
  }

  private def forwardStepModification: Actor.Receive = forwardStepModification(taskId, createOrLookupChildForTaskBlock)

  private def modifySteps: Receive = modifySteps(taskId, blockFinder)

  def createOrLookupChildForTaskBlock(): ActorRef = child("blockContainer") match {
    case Some(ref) =>
      ref
    case None =>
      createChild(PhaseContainerExecutingActor.props(task, task.getBlock, task.getContext), "blockContainer")
  }

  def notifyTaskDone() {
    info(s"Task [$taskId] is completed with state [${task.getState}]")
    context.system.eventStream.publish(TaskDone(task))
  }

  def updateStateAndNotify(newState: TaskExecutionState) {
    debug(s"Sending TaskStateEvent(${task.getState}->$newState) message for [$taskId]")
    implicit val system = context.system
    task.setTaskStateAndNotify(newState)
  }

  object ReceiveOrFail {
    def apply(receive: Receive): Receive = receive orElse {
      case m =>
        warn(s"Received unexpected message: $m in ReceiveOrFail.")
        sender() ! Failure(new IllegalStateException(s"Wrong command [$m] for task [$taskId] in state [${task.getState}]."))
    }
  }

}
