package com.xebialabs.deployit.engine.tasker

import akka.actor._
import com.xebialabs.deployit.engine.api.execution.BlockExecutionState
import com.xebialabs.deployit.engine.tasker.BlockExecutingActor.{BlockDone, BlockStateChanged}
import com.xebialabs.deployit.engine.tasker.messages.{Abort, Start, Stop}

object BlockExecutingActor {

  def props(task: Task, block: CompositeBlock, ctx: TaskExecutionContext, blockRouter: SatelliteBlockRouter) = {
    if (block.isInstanceOf[StepBlock]) throw new IllegalArgumentException("I cannot handle Step Blocks...")
    Props(classOf[BlockExecutingActor], task, block, ctx, blockRouter)
  }

  case class BlockDone(taskId: TaskId, block: Block)

  case class BlockStateChanged(taskId: TaskId, block: Block, oldState: BlockExecutionState, state: BlockExecutionState)

}

class BlockExecutingActor(val task: Task, b: CompositeBlock, ctx: TaskExecutionContext, blockRouter: SatelliteBlockRouter) extends BaseExecutionActor with SplitModifySteps with BecomeWithMdc {

  import com.xebialabs.deployit.engine.api.execution.BlockExecutionState._
  import context._

  var block: CompositeBlock = b
  val taskId = task.getId

  private def logStateChange(newState: String) = debug(s">> {STATE CHANGE}: BlockExecutingActor => [$newState]. Task: $taskId, block: $block.")

  def updateStateAndNotify(newState: BlockExecutionState) {
    val oldState: BlockExecutionState = block.state
    debug(s"updating task ${task.getId} from $oldState to $newState")
    if (oldState != newState) {
      block = block.newState(newState)
      val msg: BlockStateChanged = BlockStateChanged(task.getId, block, oldState, newState)
      debug(s"[${block.id}] : Sending BlockStateChanged($oldState->$newState) to ${parent.path}")
      parent ! msg
    } else {
      debug(s"Not sending BlockStateChanged because state is not changed. Old: $oldState, new: $newState")
    }
  }

  def receive: Actor.Receive = ReceiveWithMdc(task)(start orElse sendModifyStepsToBlocks(taskId, lookUpActor))

  private def lookUpActor(blockPath: BlockPath) = {
    block.blocks.find(_.id.isParent(blockPath)).map(createChildForBlock)
  }

  private def start: Receive = {
    logStateChange("start")
    val behaviour: Receive = {
      case Start(`taskId`) if Set(PENDING, STOPPED, ABORTED, FAILED).contains(block.state) => block match {
        case parallelBlock@ParallelBlock(id, description, _, blocks) =>
          info(s"[${block.id}] : Processing ParallelBlock[$id] of task [${task.getId}]")
          updateStateAndNotify(BlockExecutionState.EXECUTING)
          executeParallelBlock(blocks)
        case serialBlock@SerialBlock(id, description, _, blocks) =>
          info(s"[${block.id}] : Processing SerialBlock[$id] of task [${task.getId}]")
          updateStateAndNotify(BlockExecutionState.EXECUTING)
          executeSerialBlock(blocks)
      }
    }
    behaviour
  }

  def executeParallelBlock(blocks: Seq[ExecutableBlock]) {
    blocks match {
      case Seq() =>
        markAsDone
      case _ =>
        val blocksWithActors: Seq[(ExecutableBlock, ActorRef)] = blocks.filterNot(_.state == DONE).map(b => (b, createChildForBlock(b)))
        becomeWithMdc(waitForParallelBlocksCompleted(blocksWithActors))
        blocksWithActors.foreach {
          case (_, actor) => actor ! Start(taskId)
        }
    }
  }

  def executeSerialBlock(blocks: Seq[ExecutableBlock]) {
    blocks match {
      case Seq(head, tail@_*) =>
        head.state match {
          case DONE =>
            executeSerialBlock(tail)
          case _ =>
            val child: ActorRef = createChildForBlock(head)
            becomeWithMdc(waitForSerialBlockCompleted(blocks, child))
            child ! Start(taskId)
        }
      case Seq() => markAsDone
    }
  }

  private def markAsDone: Unit = {
    block.newState(DONE)
    changeState()
  }

  def createChildForBlock(b: ExecutableBlock): ActorRef = child(b.id.toBlockId) match {
    case Some(ref) => ref
    case None => b match {
      case sb: StepBlock => createChild(blockRouter.route(task, sb, executionContext(sb)), sb.id.toBlockId)
      case cb: CompositeBlock => createChild(blockRouter.route(task, cb, executionContext(cb)), cb.id.toBlockId)
    }
  }

  private def executionContext(child: ExecutableBlock) = block match {
    case s: SerialBlock => ctx
    case p: ParallelBlock => ctx.contextFor(child)
  }

  def waitForParallelBlocksCompleted(blocks: Seq[(ExecutableBlock, ActorRef)]): Actor.Receive = {
    logStateChange("waitForParallelBlocksCompleted")
    val receive: Receive = {
      case BlockDone(`taskId`, doneBlock: ExecutableBlock) =>
        debug(s"[${block.id}] : Received BlockDone message for [${doneBlock.id}]")
        val withoutCompletedBlock: Seq[(ExecutableBlock, ActorRef)] = blocks.filterNot(_._1.id == doneBlock.id)
        updateStateAndNotify(block.determineNewState(doneBlock, withoutCompletedBlock.map(_._1)))
        if (withoutCompletedBlock.isEmpty) {
          changeState()
        } else {
          becomeWithMdc(waitForParallelBlocksCompleted(withoutCompletedBlock))
        }

      case BlockStateChanged(`taskId`, changedBlock: Block, oldState, state) =>
        debug(s"[${block.id}] : Received BlockStateChanged($oldState->$state) message for [${block.id}]")
        updateStateAndNotify(block.determineNewState(changedBlock, blocks.map(_._1)))
        becomeWithMdc(waitForParallelBlocksCompleted(blocks))

      case s@Stop(`taskId`) =>
        debug(s"[${block.id}] : Received Stop($taskId) message, stopping sub-block")
        blocks.foreach(t => t._2 ! s)

      case m@Abort(`taskId`) =>
        debug(s"[${block.id}] : Received [$m] message, aborting sub-block")
        blocks.foreach(t => t._2 ! m)
    }
    receive
  }

  def waitForSerialBlockCompleted(blocks: Seq[ExecutableBlock], currentBlockActor: ActorRef): Actor.Receive = {
    logStateChange("waitForSerialBlockCompleted")
    val receive: Receive = {
      case BlockDone(`taskId`, doneBlock: Block) =>
        debug(s"[${block.id}] : Received BlockDone message for [${doneBlock.id}]")
        val withoutCompletedBlock: Seq[ExecutableBlock] = blocks.tail
        updateStateAndNotify(block.determineNewState(doneBlock, withoutCompletedBlock))
        if (withoutCompletedBlock.isEmpty || doneBlock.state != DONE) {
          changeState()
        } else {
          executeSerialBlock(withoutCompletedBlock)
        }

      case BlockStateChanged(`taskId`, changedBlock: Block, oldState, state) =>
        debug(s"[${block.id}] : Received BlockStateChange($oldState->$state) message for [${changedBlock.id}]")
        updateStateAndNotify(block.determineNewState(changedBlock, blocks))
        becomeWithMdc(waitForSerialBlockCompleted(blocks, currentBlockActor))

      case s@Stop(`taskId`) =>
        debug(s"[${block.id}] : Received Stop($taskId) message, stopping sub-block")
        currentBlockActor ! s

      case m@Abort(`taskId`) =>
        debug(s"[${block.id}] : Received [$m] message, aborting sub-block")
        currentBlockActor ! m
    }
    receive
  }

  def changeState() {
    if (block.state == DONE) {
      doNotAcceptMessages(s"[${block.id}] : All is [EXECUTED]")
    } else {
      becomeWithMdc(receive)
    }
    info(s"[${block.id}] : All sub blocks of [${block.id}] completed, notifying parent")
    parent ! BlockDone(taskId, block)
  }
}
