package com.xebialabs.deployit.engine.tasker

import akka.actor.{ActorRef, Props}
import akka.event.LoggingReceive
import com.xebialabs.deployit.engine.api.execution.BlockExecutionState
import com.xebialabs.deployit.engine.api.execution.BlockExecutionState._
import com.xebialabs.deployit.engine.tasker.BlockExecutingActor.{BlockDone, BlockStateChanged}
import com.xebialabs.deployit.engine.tasker.messages._

//Missing Schedule
class PhaseExecutingActor(task: Task, phase: Phase, ctx: TaskExecutionContext) extends BaseExecutionActor with ModifyStepsSupport {

  val taskId = task.getId

  val block = phase.block

  //Phase are always execution on xld master side
  val blockRouter = SatelliteBlockRouter.Master

  private def logStateChange(newState: String) = debug(s">> {STATE CHANGE}: PhaseExecutingActor => [$newState]. Task: $taskId, block: $block")

  override def receive: Receive = {
    logStateChange("receive")
    LoggingReceive.withLabel("receive")(
      ReceiveWithMdc(task)(start orElse forwardStepModification(taskId, createOrLookUpBlockActor))
    )
  }

  private def start: Receive = {
    logStateChange("start")
    LoggingReceive.withLabel("start") {
      case Start(`taskId`) if Set(PENDING, STOPPED, ABORTED, FAILED).contains(phase.state) =>
        info(s"[${phase.id}] : Processing Phase[${phase.id}] of task [$taskId]")
        updateStateAndNotify(EXECUTING)

        val blockActor = createOrLookUpBlockActor()

        blockActor ! Start(taskId)
        context become (waitingForBlockCompletion(blockActor) orElse forwardStepModification(taskId, createOrLookUpBlockActor))
    }
  }

  def createOrLookUpBlockActor() = context.child(block.id.toBlockId) match {
    case Some(ref) => ref
    case None => block match {
      case sb: StepBlock => createChild(blockRouter.route(task, sb, ctx), sb.id.toBlockId)
      case cb: CompositeBlock => createChild(blockRouter.route(task, cb, ctx), cb.id.toBlockId)
    }
  }

  def waitingForBlockCompletion(blockActor: ActorRef): Receive = {
    logStateChange("waitingForBlockCompletion")
    LoggingReceive.withLabel("waitingForBlockCompletion")(ReceiveWithMdc(task) {
      case BlockDone(`taskId`, `block`) =>
        debug(s"[${block.id}] : Received BlockDone message for [${block.id}]")
        updateStateAndNotify(block.state)
        notifyAndDie()

      case BlockStateChanged(`taskId`, changedBlock: Block, oldState, state) =>
        debug(s"[${block.id}] : Received BlockStateChange($oldState->$state) message for [${changedBlock.id}]")
        updateStateAndNotify(changedBlock.state)

      case stop@Stop(`taskId`) =>
        debug(s"[${block.id}] : Received Stop($taskId) message, stopping sub-block")
        blockActor ! stop

      case abort@Abort(`taskId`) =>
        debug(s"[${block.id}] : Received [$abort] message, aborting sub-block")
        blockActor ! abort
    })
  }

  def notifyAndDie() {
    if (block.state == DONE) {
      doNotAcceptMessages(s"[${block.id}] : All is [EXECUTED]")
    } else {
      context become receive
    }

    info(s"[${block.id}] : All sub blocks of [${block.id}] completed with state ${phase.state}, notifying parent")
    context.parent ! BlockDone(taskId, phase)
  }

  def updateStateAndNotify(newState: BlockExecutionState) {
    val oldState: BlockExecutionState = phase.state
    if (oldState != newState) {
      phase.newState(newState)
      val msg = BlockStateChanged(task.getId, phase, oldState, newState)
      debug(s"[${phase.id}] : Sending BlockStateChanged($oldState->$newState) to ${context.parent.path}")
      context.parent ! msg
    }
  }
}

object PhaseExecutingActor {
  def props(task: Task, phase: Phase, ctx: TaskExecutionContext): Props =
    Props(classOf[PhaseExecutingActor], task, phase, ctx)
}