package com.xebialabs.deployit.engine.tasker

import akka.actor.Status.Failure
import akka.actor._
import com.github.nscala_time.time.Imports._
import com.xebialabs.deployit.engine.api.execution.TaskExecutionState._
import com.xebialabs.deployit.engine.api.execution.{BlockExecutionState => BlockIs, TaskExecutionState}
import com.xebialabs.deployit.engine.tasker.ArchiveActor.messages.SendToArchive
import com.xebialabs.deployit.engine.tasker.BlockExecutingActor.{BlockDone, BlockStateChanged}
import com.xebialabs.deployit.engine.tasker.StateChangeEventListenerActor.{StepStateEvent, TaskStateEvent}
import com.xebialabs.deployit.engine.tasker.TaskManagingActor.messages.{ArchiveTask, Cancel, Recovered, Register, Schedule, ScheduledStart}
import com.xebialabs.deployit.engine.tasker.messages._
import org.joda.time.DateTime

import scala.concurrent.duration._

object TaskManagingActor {
  def props(task: Task) = Props(classOf[TaskManagingActor], task)

  def getMessages = messages

  object messages {

    case class Register(task: Task)

    case class ScheduledStart(taskId: TaskId)

    case class Schedule(taskId: TaskId, scheduleAt: DateTime)

    case class ArchiveTask(taskId: TaskId, archiveActor: ActorRef, notificationActor: ActorRef)

    case class Cancel(taskId: TaskId, archiveActor: ActorRef, notificationActor: ActorRef)

    case class Recovered(task: Task)

  }

}

class TaskManagingActor(task: Task) extends BaseExecutionActor with Stash with ModifyStepsSupport {

  import context._

  val taskId = task.getId

  val terminationStates = Set(FAILED, ABORTED, EXECUTED, STOPPED)

  private val blockFinder: BlockFinder = task.getBlock

  def receive: Actor.Receive = ReceiveWithMdc() {
    case Register(`task`) =>
      info(s"Received [Register] message for task [$taskId]")
      registerStateListeners(task)

      updateStateAndNotify(PENDING)
      become(pending)

      TaskRegistryExtension(system).register(task)
      sender() ! Registered(taskId)
    case Recovered(`task`) =>
      info(s"Received [Recovered] message for task [$taskId]")
      registerStateListeners(task)
      task.getState match {
        case PENDING | QUEUED =>
          become(pending)
        case SCHEDULED if task.getScheduledDate.isAfterNow =>
          doSchedule(task.getScheduledDate, createOrLookupChildForTaskBlock())
        case SCHEDULED =>
          doEnqueue(createOrLookupChildForTaskBlock())
        case EXECUTED =>
          become(canBeArchived)
        case STOPPED | FAILED | ABORTED =>
          become(readyForRestart(createOrLookupChildForTaskBlock()))
        case _ =>
          sender() ! Failure(new IllegalStateException(s"Cannot recover a task which is in state [${task.getState}]."))
      }
      TaskRegistryExtension(system).register(task)
      sender() ! Registered(taskId)
  }

  def pending: Actor.Receive = modifySteps orElse ReceiveOrFail {
    case Enqueue(`taskId`) =>
      doEnqueue(createOrLookupChildForTaskBlock())
    case Schedule(`taskId`, scheduleAt) =>
      doSchedule(scheduleAt, createOrLookupChildForTaskBlock())
    case Cancel(`taskId`, archiveActor, notificationActor) =>
      doCancelWhenPending(notificationActor)
  }

  def queued(blockActor: ActorRef): Actor.Receive = ReceiveOrFail {
    case Start(`taskId`) =>
      doStart(blockActor)
  }

  def scheduled(scheduleHandle: Cancellable, blockActor: ActorRef): Actor.Receive = ReceiveWithMdc(task) {
    case Enqueue(`taskId`) =>
      scheduleHandle.cancel()
      doEnqueue(blockActor)
    case Schedule(`taskId`, scheduleAt) =>
      scheduleHandle.cancel()
      doSchedule(scheduleAt, blockActor)
    case ScheduledStart(`taskId`) =>
      doEnqueue(blockActor)
    case Cancel(`taskId`, archiveActor, notificationActor) if task.getStartDate == null =>
      scheduleHandle.cancel()
      doCancelWhenPending(notificationActor)
    case Cancel(`taskId`, archiveActor, notificationActor) if task.getStartDate != null =>
      scheduleHandle.cancel()
      doCancel(archiveActor, notificationActor)
    case _ => throw new IllegalStateException()
  }

  def executing(blockActor: ActorRef): Actor.Receive = ReceiveOrFail {
    case BlockStateChanged(`taskId`, block, oldState, state) =>
      state match {
        case BlockIs.PENDING | BlockIs.EXECUTING =>
        case BlockIs.DONE =>
          updateStateAndNotify(EXECUTED)
          become(executed)
        case BlockIs.FAILING =>
          updateStateAndNotify(FAILING)
          become(failing(blockActor, readyForRestart(blockActor)))
        case BlockIs.FAILED =>
          updateStateAndNotify(FAILED)
          become(failed(blockActor, forwardStepModification orElse readyForRestart(blockActor)))
        case BlockIs.STOPPING =>
          updateStateAndNotify(STOPPING)
          become(stopping(blockActor, readyForRestart(blockActor)))
        case BlockIs.STOPPED =>
          updateStateAndNotify(STOPPED)
          become(stopped(blockActor, forwardStepModification orElse readyForRestart(blockActor)))
        case BlockIs.ABORTING =>
          updateStateAndNotify(ABORTING)
          become(aborting(blockActor, readyForRestart(blockActor)))
        case BlockIs.ABORTED =>
          updateStateAndNotify(ABORTED)
          become(aborted(blockActor, forwardStepModification orElse readyForRestart(blockActor)))
      }
    case s@Stop(`taskId`) =>
      debug(s"Received [$s], now stopping execution")
      blockActor ! s
    case m@Abort(`taskId`) =>
      debug(s"Received [$m], now aborting execution")
      blockActor ! m
  }

  def executed: Actor.Receive = ReceiveOrFail {
    case BlockDone(`taskId`, block) =>
      task.recordCompletion()
      notifyTaskDone()
      become(canBeArchived)
      unstashAll()
    case ArchiveTask(`taskId`, _, _) => stash()
  }

  def canBeArchived: Actor.Receive = ReceiveOrFail {
    case ArchiveTask(`taskId`, archiveActor: ActorRef, notificationActor: ActorRef) =>
      info(s"Received [Archive] message for task [$taskId]")
      context.system.eventStream.subscribe(self, classOf[TaskStateEventHandled])
      TaskRegistryExtension(system).deleteTask(taskId)
      updateStateAndNotify(DONE)
      become(waitForStateHandledThenArchive(DONE, archiveActor, notificationActor))
  }

  def archiving(notificationActor: ActorRef): Actor.Receive = ReceiveWithMdc(task) {
    case Archived(`taskId`) =>
      notifyTaskDone()
      task.getState match {
        case DONE => notificationActor ! Archived(taskId)
        case CANCELLED => notificationActor ! Cancelled(taskId)
        case _ @ state => debug(s"Unexpected state: $state for task: $taskId after task archive operation.")
      }
      harakiri(s"Done with task [$taskId]")

    case fta@FailedToArchive(`taskId`, exception) =>
      info(s"Task [$taskId] failed to archive, going back to previous state.")
      task.getState match {
        case DONE =>
          updateStateAndNotify(EXECUTED)
          become(executed)
        case CANCELLED =>
          updateStateAndNotify(FAILED)
          val blockActor = createOrLookupChildForTaskBlock()
          become(failed(blockActor, readyForRestart(blockActor)))
        case _ @ state => debug(s"Unexpected state: $state for task: $taskId when failed to archive.")
      }
      notifyTaskDone()
      notificationActor ! fta
    case _ @ message => debug(s"Unexpected message during task: $taskId is being archived.")
  }


  def failing(blockActor: ActorRef, restartState: Receive): Actor.Receive = ReceiveOrFail {
    case BlockStateChanged(`taskId`, block, oldState, state) =>
      state match {
        case BlockIs.PENDING | BlockIs.EXECUTING | BlockIs.DONE | BlockIs.FAILING | BlockIs.STOPPING | BlockIs.STOPPED =>
        case BlockIs.FAILED =>
          updateStateAndNotify(FAILED)
          become(failed(blockActor, forwardStepModification orElse restartState))
        case BlockIs.ABORTING =>
          updateStateAndNotify(ABORTING)
          become(aborting(blockActor, restartState))
        case BlockIs.ABORTED =>
          updateStateAndNotify(ABORTED)
          become(aborted(blockActor, forwardStepModification orElse restartState))
      }
    case s@Stop(`taskId`) =>
      debug(s"Received [$s], now stopping execution")
      blockActor ! s
    case m@Abort(`taskId`) =>
      debug(s"Received [$m], now aborting execution")
      blockActor ! m
  }


  def failed(blockActor: ActorRef, restartState: Receive): Actor.Receive = ReceiveOrFail {
    case BlockDone(`taskId`, block) =>
      task.recordFailure()
      task.recordCompletion()
      notifyTaskDone()
      become(restartState)
      unstashAll()
    case Enqueue(`taskId`) | Schedule(`taskId`, _) | Cancel(`taskId`, _, _) => stash()
  }

  def readyForRestart(blockActor: ActorRef): Actor.Receive = modifySteps orElse ReceiveOrFail {
    case Enqueue(`taskId`) =>
      doEnqueue(blockActor)

    case Schedule(`taskId`, scheduleAt) =>
      doSchedule(scheduleAt, blockActor)

    case Cancel(`taskId`, archiveActor, notificationActor) =>
      doCancel(archiveActor, notificationActor)
  }

  def cancelling(blockActor: ActorRef, notificationActor: ActorRef, waitForStateHandle: Receive, readyState: Receive): Actor.Receive = ReceiveOrFail {
    case BlockStateChanged(`taskId`, block, oldState, state) =>
      state match {
        case BlockIs.PENDING | BlockIs.EXECUTING =>
        case BlockIs.DONE =>
          context.system.eventStream.subscribe(self, classOf[TaskStateEventHandled])
          updateStateAndNotify(CANCELLED)
          notifyTaskDone()
          TaskRegistryExtension(system).deleteTask(taskId)
          become(waitForStateHandle)
        case BlockIs.FAILING =>
          updateStateAndNotify(FAILING)
          become(failing(blockActor, readyState))
        case BlockIs.FAILED =>
          updateStateAndNotify(FAILED)
          become(failed(blockActor, forwardStepModification orElse readyState))
        case BlockIs.STOPPING =>
          updateStateAndNotify(STOPPING)
          become(stopping(blockActor, readyState))
        case BlockIs.STOPPED =>
          updateStateAndNotify(STOPPED)
          become(stopped(blockActor, forwardStepModification orElse readyState))
        case BlockIs.ABORTING =>
          updateStateAndNotify(ABORTING)
          become(aborting(blockActor, readyState))
        case BlockIs.ABORTED =>
          updateStateAndNotify(ABORTED)
          become(aborted(blockActor, forwardStepModification orElse readyState))

      }
    case s@Stop(`taskId`) =>
      debug(s"Received [$s], now stopping execution")
      blockActor ! s
    case m@Abort(`taskId`) =>
      debug(s"Received [$m], now aborting execution")
      blockActor ! m
  }

  def readyForCancelling(archiveActor: ActorRef, notificationActor: ActorRef): Receive = {
    case Enqueue(`taskId`) =>
      doCancel(archiveActor, notificationActor)
  }

  def readyForCancellingWithoutArchive(notificationActor: ActorRef): Receive = {
    case Enqueue(`taskId`) =>
      doCancelWhenPending(notificationActor)
  }

  def waitForStateHandledThenArchive(state: TaskExecutionState, archiveActor: ActorRef, notificationActor: ActorRef, messagesToHandle: Int = 2): Actor.Receive = ReceiveWithMdc(task) {
    case TaskStateEventHandled(`taskId`, _, `state`) if messagesToHandle == 1 && !task.getSpecification.isArchiveable =>
      become(archiving(notificationActor))
      self ! Archived(task.getId)
    case TaskStateEventHandled(`taskId`, _, `state`) if messagesToHandle == 1 =>
      become(archiving(notificationActor))
      archiveActor ! SendToArchive(task, self)
    case TaskStateEventHandled(`taskId`, _, `state`) => become(waitForStateHandledThenArchive(state, archiveActor, notificationActor, messagesToHandle - 1))
    case _ @ message => debug(s"Unexpected message: {$message} when waiting for state handled before archive. Messages to handle: $messagesToHandle.")
  }

  def stopping(blockActor: ActorRef, restartState: Receive): Actor.Receive = ReceiveOrFail {
    case BlockStateChanged(`taskId`, block, oldState, state) =>
      state match {
        case BlockIs.PENDING | BlockIs.EXECUTING | BlockIs.STOPPING =>
        case BlockIs.DONE =>
          updateStateAndNotify(EXECUTED)
          become(executed)
        case BlockIs.FAILING =>
          updateStateAndNotify(FAILING)
          become(failing(blockActor, restartState))
        case BlockIs.FAILED =>
          updateStateAndNotify(FAILED)
          become(failed(blockActor, forwardStepModification orElse restartState))
        case BlockIs.STOPPED =>
          updateStateAndNotify(STOPPED)
          become(stopped(blockActor, forwardStepModification orElse restartState))
        case BlockIs.ABORTING =>
          updateStateAndNotify(ABORTING)
          become(aborting(blockActor, restartState))
        case BlockIs.ABORTED =>
          updateStateAndNotify(ABORTED)
          become(aborted(blockActor, forwardStepModification orElse restartState))
      }
    case s@Stop(`taskId`) =>
      debug(s"Received [$s], now stopping execution")
      blockActor ! s
    case m@Abort(`taskId`) =>
      debug(s"Received [$m], now aborting execution")
      blockActor ! m
  }

  def stopped(blockActor: ActorRef, restartState: Receive): Actor.Receive = ReceiveOrFail {
    case m@BlockDone(`taskId`, block) =>
      debug(s"stopped(), received $m")
      task.recordCompletion()
      notifyTaskDone()
      become(restartState)
      unstashAll()
    case Enqueue(`taskId`) | Schedule(`taskId`, _) | Cancel(`taskId`, _, _) => stash()
  }

  def aborting(blockActor: ActorRef, restartState: Receive): Actor.Receive = ReceiveOrFail {
    case BlockStateChanged(`taskId`, block, oldState, state) =>
      state match {
        case BlockIs.PENDING | BlockIs.EXECUTING | BlockIs.DONE | BlockIs.FAILING | BlockIs.STOPPING | BlockIs.STOPPED | BlockIs.FAILED | BlockIs.ABORTING =>
        case BlockIs.ABORTED =>
          updateStateAndNotify(ABORTED)
          become(aborted(blockActor, forwardStepModification orElse restartState))
      }

  }

  def aborted(blockActor: ActorRef, restartState: Receive): Actor.Receive = ReceiveOrFail {
    case m@BlockDone(`taskId`, block) =>
      debug(s"aborted(), received $m")
      task.recordFailure()
      task.recordCompletion()
      notifyTaskDone()
      become(restartState)
      unstashAll()
    case Enqueue(`taskId`) | Schedule(`taskId`, _) | Cancel(`taskId`, _, _) => stash()
  }

  def registerStateListeners(task: Task) {
    def registerStateListener(child: ActorRef) {
      context.system.eventStream.subscribe(child, classOf[TaskStateEvent])
      context.system.eventStream.subscribe(child, classOf[StepStateEvent])
    }

    val child: ActorRef = createChild(StateChangeEventListenerActor.props(taskId), "state-listener")
    registerStateListener(child)
  }

  def calculateDelay(time: DateTime): FiniteDuration = FiniteDuration((DateTime.now to time).millis, MILLISECONDS)

  def doEnqueue(blockActor: ActorRef) {
    info(s"Received [Enqueue] message for task [$taskId]")
    updateStateAndNotify(QUEUED)
    become(queued(blockActor))
    self ! Start(taskId)
  }

  def doStart(blockActor: ActorRef) {
    info(s"Received [Start] message for task [$taskId]")
    task.recordStart()
    updateStateAndNotify(EXECUTING)
    become(executing(blockActor))
    blockActor ! Start(taskId)

    info(s"sending ${Start(taskId)} to $blockActor")
  }

  def doSchedule(scheduleAt: DateTime, blockActor: ActorRef) {
    info(s"Received [Schedule] message for task [$taskId]")
    task.setScheduledDate(scheduleAt)
    updateStateAndNotify(SCHEDULED)

    val delay: FiniteDuration = calculateDelay(scheduleAt)
    info(s"Going to schedule task [$taskId] at [${scheduleAt.toString("yyyy-MM-dd hh:mm:ss Z")}] which is [$delay] from now")

    if (delay.toMinutes < 0) {
      doEnqueue(blockActor)
    } else {
      val scheduleHandle: Cancellable = context.system.scheduler.scheduleOnce(delay, self, Enqueue(taskId))
      become(scheduled(scheduleHandle, blockActor))
    }
  }

  def doCancelWhenPending(notificationActor: ActorRef) {
    info(s"Received [Cancel] message for task [$taskId]")
    context.system.eventStream.subscribe(self, classOf[TaskStateEventHandled])
    updateStateAndNotify(CANCELLED)
    notifyTaskDone()
    TaskRegistryExtension(system).deleteTask(taskId)
    var messagesLeftCounter = 2
    become({
      case TaskStateEventHandled(`taskId`, _, CANCELLED) if messagesLeftCounter == 1 =>
        notificationActor ! Cancelled(taskId)
        context.system.eventStream.unsubscribe(self)
      case TaskStateEventHandled(`taskId`, _, CANCELLED) => messagesLeftCounter -= 1
      case _ @ message => debug(s"Unexpected message: {$message} while waiting for cancelled state event. messagesLeftCounter: $messagesLeftCounter")
    })
  }

  def waitForCancelWithoutArchive(notificationActor: ActorRef, handledMessagesCounter: Int): Receive = {
    case TaskStateEventHandled(`taskId`, _, CANCELLED) if handledMessagesCounter == 1 =>
      notificationActor ! Cancelled(taskId)
      context.system.eventStream.unsubscribe(self)
    case TaskStateEventHandled(`taskId`, _, CANCELLED) => become(waitForCancelWithoutArchive(notificationActor, handledMessagesCounter - 1))
    case _ @ message => debug(s"Unexpected message: {$message} while waiting for cancel without archiving. handledMessagesCounter: $handledMessagesCounter")
  }

  def doCancel(archiveActor: ActorRef, notificationActor: ActorRef) {
    info(s"Received [Cancel] message for task [$taskId]")

    val blockActor = createOrLookupChildForTaskBlock()
    blockActor ! FinishUp(taskId)

    updateStateAndNotify(CANCELLING)

    val waitForStateHandled = waitForStateHandledThenArchive(CANCELLED, archiveActor, notificationActor)
    val readyState = readyForCancelling(archiveActor, notificationActor)

    become(cancelling(blockActor, notificationActor, waitForStateHandled, readyState))
  }

  private def forwardStepModification: Actor.Receive = forwardStepModification(taskId, createOrLookupChildForTaskBlock)

  private def modifySteps: Receive = modifySteps(taskId, blockFinder)

  def createOrLookupChildForTaskBlock(): ActorRef = child("blockContainer") match {
    case Some(ref) =>
      ref
    case None =>
      createChild(PhaseContainerExecutingActor.props(task, task.getBlock, task.getContext), "blockContainer")
  }

  def notifyTaskDone() {
    info(s"Task [$taskId] is completed with state [${task.getState}]")
    context.system.eventStream.publish(TaskDone(task))
  }

  def updateStateAndNotify(newState: TaskExecutionState) {
    debug(s"Sending TaskStateEvent(${task.getState}->$newState) message for [$taskId]")
    implicit val system = context.system
    task.setTaskStateAndNotify(newState)
  }

  object ReceiveOrFail {
    def apply(receive: Receive): Receive = ReceiveWithMdc(task)(receive) orElse {
      case m => sender() ! Failure(new IllegalStateException(s"Wrong command [$m] for task [$taskId] in state [${task.getState}]."))
    }
  }

}
