package com.xebialabs.deployit.engine.tasker

import akka.actor.{Actor, ActorRef, Props}
import com.xebialabs.deployit.engine.api.execution.BlockExecutionState
import com.xebialabs.deployit.engine.api.execution.BlockExecutionState._
import com.xebialabs.deployit.engine.tasker.BlockExecutingActor.{BlockDone, BlockStateChanged}
import com.xebialabs.deployit.engine.tasker.messages._

object PhaseContainerExecutingActor {
  def props(task: Task, block: PhaseContainer, ctx: TaskExecutionContext) = {
    Props(classOf[PhaseContainerExecutingActor], task, block, ctx)
  }

}

class PhaseContainerExecutingActor(val task: Task, container: PhaseContainer, ctx: TaskExecutionContext) extends BaseExecutionActor with ActorCreationSupport with SplitModifySteps with BecomeWithMdc {

  val taskId = task.getId

  override def receive: Receive = sendModifyStepsToBlocks(taskId, lookUpPhaseActor) orElse ReceiveWithMdc(task) {
    case Start(`taskId`) if Set(PENDING, STOPPED, ABORTED, FAILED).contains(container.state) =>
      info(s"[${container.id}] : Processing PhaseContainer of task [${task.getId}]")
      updateStateAndNotify(BlockExecutionState.EXECUTING)
      executePhases(container.phases)

    case Start(`taskId`) =>
      error(s"can't start a container in stake ${container.state}")

    case finishUp@FinishUp(`taskId`) =>
      debug(s"[${container.id}] : Received [$finishUp] message, execute finish up phase")

      val alwaysExecutedPhases = container.phases
        .collect {
        case Phase.AlwaysExecuted(forcedPhase) => forcedPhase
      }.dropWhile(phase => phase.getState() == DONE)
        .toList

      alwaysExecutedPhases match {
        case alwaysExecutedPhase :: tail =>
          task.cancelling = true
          updateStateAndNotify(BlockExecutionState.EXECUTING)
          val phaseActor = createOrLookUpPhaseActor(alwaysExecutedPhase)
          phaseActor ! Start(taskId)
          becomeWithMdc(waitForPhaseCompletion(alwaysExecutedPhase, alwaysExecutedPhases, phaseActor))
        case Nil =>
          updateStateAndNotify(DONE)
          notifyParent()
      }
  }

  private def lookUpPhaseActor(blockPath: BlockPath) = {
    container.phases.find(_.id.isParent(blockPath)).map(p => createOrLookUpPhaseActor(p))
  }

  def updateStateAndNotify(newState: BlockExecutionState) {
    val oldState = container.state
    debug(s"Checking to update state from $oldState -> $newState")
    if (oldState != newState) {
      container.newState(newState)
      val msg = BlockStateChanged(task.getId, container, oldState, newState)
      debug(s"[${container.id}] : Sending BlockStateChanged($oldState->$newState) to ${context.parent.path}")
      context.parent ! msg
    }
  }

  def executePhases(phases: Seq[Phase]) {
    val phase = phases.head
    phase.state match {
      case DONE =>
        executePhases(phases.tail)
      case _ =>
        val phaseActor = createOrLookUpPhaseActor(phase)
        phaseActor ! Start(taskId)
        becomeWithMdc(waitForPhaseCompletion(phase, phases, phaseActor))
    }
  }

  def createOrLookUpPhaseActor(phase: Phase): ActorRef =
    context.child(phase.getId()) match {
      case Some(ref) =>
        ref
      case None =>
        createChild(Props(new PhaseExecutingActor(task, phase, ctx)), phase.getId())
    }

  def waitForPhaseCompletion(phase: Phase, phases: Seq[Phase], phaseActor: ActorRef): Actor.Receive = {

    case m@BlockDone(`taskId`, `phase`) =>
      debug(s"[${container.id}] : Received $m message for [${phase.id}]")
      val remainingPhases = phases.tail
      val newState: BlockExecutionState = container.determineNewState(phase, remainingPhases)
      debug(s"Determined new container state to be [$newState] for $phase and $remainingPhases")
      updateStateAndNotify(newState)
      if (newState.isFinished) {
        notifyParent()
      } else {
        executePhases(remainingPhases)
      }

    case m@BlockStateChanged(`taskId`, `phase`, oldState, state) =>
      debug(s"[${container.id}] : Received $m message for [${phase.id}]")
      val newState: BlockExecutionState = container.determineNewState(phase, phases)
      debug(s"Determined new container state to be [$newState] for $phase and $phases")
      updateStateAndNotify(newState)

    case stop@Stop(`taskId`) =>
      debug(s"[${container.id}] : Received Stop($taskId) message, stopping sub-block")
      phaseActor ! stop

    case abort@Abort(`taskId`) =>
      debug(s"[${container.id}] : Received [$abort] message, aborting sub-block")
      phaseActor ! abort
  }

  def notifyParent() {
    if (container.state != DONE) {
      context become receive
    } else {
      doNotAcceptMessages("Container DONE")
    }
    info(s"[${container.id}] : All phases of [${container.id}] completed, notifying parent")
    context.parent ! BlockDone(taskId, container)

  }
}
