package com.xebialabs.deployit.engine.tasker.satellite

import java.util.zip.{Deflater, Inflater}

import akka.actor.ExtendedActorSystem
import akka.event.Logging
import akka.serialization.Serializer
import com.esotericsoftware.kryo.Kryo
import com.esotericsoftware.kryo.io.{Input, Output}
import net.jpountz.lz4.LZ4Factory

import scala.collection.mutable
import scala.util.{Failure, Success, Try}


//copied from kryo but with our magic combination of Kryo libs
//@see https://github.com/romix/akka-kryo-serialization
trait KryoCompressor {
  def compress(inputBuff: Array[Byte]): Array[Byte]

  def decompress(inputBuff: Array[Byte]): Array[Byte]
}

class NoKryoComressor extends KryoCompressor {
  def compress(inputBuff: Array[Byte]) = inputBuff

  def decompress(inputBuff: Array[Byte]) = inputBuff
}

class LZ4KryoComressor extends KryoCompressor {

  lazy val lz4factory = LZ4Factory.fastestInstance

  def compress(inputBuff: Array[Byte]): Array[Byte] = {
    val inputSize = inputBuff.length
    val lz4 = lz4factory.fastCompressor
    val maxOutputSize = lz4.maxCompressedLength(inputSize)
    val outputBuff = new Array[Byte](maxOutputSize + 4)
    val outputSize = lz4.compress(inputBuff, 0, inputSize, outputBuff, 4, maxOutputSize)

    // encode 32 bit lenght in the first bytes
    outputBuff(0) = (inputSize & 0xff).toByte
    outputBuff(1) = (inputSize >> 8 & 0xff).toByte
    outputBuff(2) = (inputSize >> 16 & 0xff).toByte
    outputBuff(3) = (inputSize >> 24 & 0xff).toByte
    outputBuff.take(outputSize + 4)
  }

  def decompress(inputBuff: Array[Byte]): Array[Byte] = {
    // the first 4 bytes are the original size
    val size: Int = (inputBuff(0).asInstanceOf[Int] & 0xff) |
      (inputBuff(1).asInstanceOf[Int] & 0xff) << 8 |
      (inputBuff(2).asInstanceOf[Int] & 0xff) << 16 |
      (inputBuff(3).asInstanceOf[Int] & 0xff) << 24
    val lz4 = lz4factory.fastDecompressor()
    val outputBuff = new Array[Byte](size)
    lz4.decompress(inputBuff, 4, outputBuff, 0, size)
    outputBuff
  }
}

class ZipKryoComressor extends KryoCompressor {

  lazy val deflater = new Deflater(Deflater.BEST_SPEED)
  lazy val inflater = new Inflater()

  def compress(inputBuff: Array[Byte]): Array[Byte] = {
    val inputSize = inputBuff.length
    val outputBuff = new mutable.ArrayBuilder.ofByte
    outputBuff += (inputSize & 0xff).toByte
    outputBuff += (inputSize >> 8 & 0xff).toByte
    outputBuff += (inputSize >> 16 & 0xff).toByte
    outputBuff += (inputSize >> 24 & 0xff).toByte

    deflater.setInput(inputBuff)
    deflater.finish()
    val buff = new Array[Byte](4096)

    while (!deflater.finished) {
      val n = deflater.deflate(buff)
      outputBuff ++= buff.take(n)
    }
    deflater.reset()
    outputBuff.result()
  }

  def decompress(inputBuff: Array[Byte]): Array[Byte] = {
    val size: Int = (inputBuff(0).asInstanceOf[Int] & 0xff) |
      (inputBuff(1).asInstanceOf[Int] & 0xff) << 8 |
      (inputBuff(2).asInstanceOf[Int] & 0xff) << 16 |
      (inputBuff(3).asInstanceOf[Int] & 0xff) << 24
    val outputBuff = new Array[Byte](size)
    inflater.setInput(inputBuff, 4, inputBuff.length - 4)
    inflater.inflate(outputBuff)
    inflater.reset()
    outputBuff
  }
}

class KryoAkkaSerializer(val system: ExtendedActorSystem) extends Serializer {

  import com.romix.akka.serialization.kryo.KryoSerialization._

  val log = Logging(system, getClass.getName)

  val settings = new Settings(system.settings.config)

  val mappings = settings.ClassNameMappings

  locally {
    log.debug("Got mappings: {}", mappings)
  }

  val classnames = settings.ClassNames

  locally {
    log.debug("Got classnames for incremental strategy: {}", classnames)
  }

  val bufferSize = settings.BufferSize

  locally {
    log.debug("Got buffer-size: {}", bufferSize)
  }

  val serializerPoolSize = settings.SerializerPoolSize

  val idStrategy = settings.IdStrategy

  locally {
    log.debug("Got id strategy: {}", idStrategy)
  }

  val serializerType = settings.SerializerType

  locally {
    log.debug("Got serializer type: {}", serializerType)
  }

  val implicitRegistrationLogging = settings.ImplicitRegistrationLogging
  locally {
    log.debug("Got implicit registration logging: {}", implicitRegistrationLogging)
  }

  val useManifests = settings.UseManifests
  locally {
    log.debug("Got use manifests: {}", useManifests)
  }

  val customSerializerInitClassName = settings.KryoCustomSerializerInit
  locally {
    log.debug("Got custom serializer init class: {}", customSerializerInitClassName)
  }

  val customSerializerInitClass: Some[Class[_ <: AnyRef]] =
    if (customSerializerInitClassName == null) null
    else
      system.dynamicAccess.getClassFor[AnyRef](customSerializerInitClassName) match {
        case Success(clazz) => Some(clazz)
        case Failure(e) =>
          log.error("Class could not be loaded and/or registered: {} ", customSerializerInitClassName)
          throw e
      }

  locally {
    log.debug("Got serializer init class: {}", customSerializerInitClass)
  }

  val customizerInstance = Try(customSerializerInitClass.map(_.newInstance))
  locally {
    log.debug("Got customizer instance: {}", customizerInstance)
  }

  val customizerMethod = Try(customSerializerInitClass.map(_.getMethod("customize", classOf[Kryo])))

  locally {
    log.debug("Got customizer method: {}", customizerMethod)
  }
  val compressor: KryoCompressor = settings.Compression match {
    case "lz4" => new LZ4KryoComressor
    case "deflate" => new ZipKryoComressor
    case _ => new NoKryoComressor
  }
  locally {
    log.debug("Got compression: {}", settings.Compression)
  }

  val serializer = try new KryoBasedSerializer(KryoSerializer.getMagicCombination(system),
    bufferSize,
    serializerPoolSize,
    useManifests)
  catch {
    case e: Exception =>
      log.error("exception caught during akka-kryo-serialization startup: {}", e)
      throw e
  }

  locally {
    log.debug("Got serializer: {}", serializer)
  }

  // This is whether "fromBinary" requires a "clazz" or not
  def includeManifest: Boolean = useManifests

  // A unique identifier for this Serializer
  def identifier = 123454323

  // Delegate to a real serializer
  def toBinary(obj: AnyRef): Array[Byte] = {
    val ser = getSerializer
    val bin = ser.toBinary(obj)
    releaseSerializer(ser)
    compressor.compress(bin)
  }

  def fromBinary(bytes: Array[Byte], clazz: Option[Class[_]]): AnyRef = {
    val ser = getSerializer
    val obj = ser.fromBinary(compressor.decompress(bytes), clazz)
    releaseSerializer(ser)
    obj
  }

  val serializerPool = new ObjectPool[Serializer](serializerPoolSize, () => {
    new KryoBasedSerializer(KryoSerializer.getMagicCombination(system),
      bufferSize,
      serializerPoolSize,
      useManifests)
  })

  private def getSerializer = serializerPool.fetch()

  private def releaseSerializer(ser: Serializer) = serializerPool.release(ser)


}

/**
 * *
 * Kryo-based serializer backend
 */
class KryoBasedSerializer(val kryo: Kryo, val bufferSize: Int, val bufferPoolSize: Int, val useManifests: Boolean) extends Serializer {

  // This is whether "fromBinary" requires a "clazz" or not
  def includeManifest: Boolean = useManifests

  // A unique identifier for this Serializer
  def identifier = 12454323

  // "toBinary" serializes the given object to an Array of Bytes
  def toBinary(obj: AnyRef): Array[Byte] = {
    val buffer = getBuffer
    try {
      if (!useManifests)
        kryo.writeClassAndObject(buffer, obj)
      else
        kryo.writeObject(buffer, obj)
      buffer.toBytes
    } finally
      releaseBuffer(buffer)
  }

  // "fromBinary" deserializes the given array,
  // using the type hint (if any, see "includeManifest" above)
  // into the optionally provided classLoader.
  def fromBinary(bytes: Array[Byte], clazz: Option[Class[_]]): AnyRef = {
    if (!useManifests)
      kryo.readClassAndObject(new Input(bytes))
    else {
      clazz match {
        case Some(c) => kryo.readObject(new Input(bytes), c).asInstanceOf[AnyRef]
        case _ => throw new RuntimeException("Object of unknown class cannot be deserialized")
      }
    }
  }

  val buf = new Output(bufferSize, 1024 * 1024)

  private def getBuffer = buf

  private def releaseBuffer(buffer: Output) = {
    buffer.clear()
  }

}

import java.util.concurrent.atomic.AtomicInteger
import java.util.concurrent.{ArrayBlockingQueue, TimeUnit}

// Support pooling of objects. Useful if you want to reduce
// the GC overhead and memory pressure.
class ObjectPool[T](number: Int, newInstance: () => T) {

  private val size = new AtomicInteger(0)
  private val pool = new ArrayBlockingQueue[T](number)

  def fetch(): T = {
    pool.poll() match {
      case o if o != null => o
      case null => createOrBlock
    }
  }

  def release(o: T): Unit = {
    pool.offer(o)
  }

  def add(o: T): Unit = {
    pool.add(o)
  }

  private def createOrBlock: T = {
    size.get match {
      case e: Int if e == number => block
      case _ => create
    }
  }

  private def create: T = {
    size.incrementAndGet match {
      case e: Int if e > number =>
        size.decrementAndGet; fetch()
      case e: Int => newInstance()
    }
  }

  private def block: T = {
    val timeout = 5000
    pool.poll(timeout, TimeUnit.MILLISECONDS) match {
      case o if o != null => o
      case _ => throw new Exception("Couldn't acquire object in %d milliseconds.".format(timeout))
    }
  }
}
