package com.xebialabs.deployit.engine.tasker

import java.io.{PrintWriter, Serializable, StringWriter}
import java.util

import com.xebialabs.deployit.engine.spi.execution.ExecutionStateListener
import com.xebialabs.deployit.plugin.api.execution.ExecutionContextListener
import com.xebialabs.deployit.plugin.api.flow.{ExecutionContext, ITask}
import com.xebialabs.deployit.plugin.api.inspection.InspectionContext
import com.xebialabs.deployit.plugin.api.services.Repository
import grizzled.slf4j.Logger


class TaskExecutionContext(parent: Option[TaskExecutionContext], spec: TaskSpecification) extends Serializable {
  private[this] val attributes: Cache[String, AnyRef] = Cache()

  attributes += (TaskExecutionContext.CACHE_KEY -> this.spec)

  @transient var repository: Repository = null

  if (spec != null) {
    import scala.collection.convert.wrapAll._
    spec.getListeners.foreach(l => attributes.put(l.getClass.getName, l))
  }

  private def getParent: Option[TaskExecutionContext] = parent

  val childContexts: Cache[BlockPath, TaskExecutionContext] = Cache()

  def contextFor(block: Block): TaskExecutionContext = contextFor(block.id)

  def contextFor(path: BlockPath): TaskExecutionContext = childContexts.getOrElse(path, {
    val c = new TaskExecutionContext(Some(this), spec)
    childContexts.put(path, c)
    c
  })

  private[tasker] def allListeners: Set[ExecutionStateListener] = {
    def getRootContext(tec: TaskExecutionContext): TaskExecutionContext = tec.getParent match {
      case Some(pp) => getRootContext(pp)
      case None => tec
    }

    def collectListeners(t: TaskExecutionContext): Set[ExecutionStateListener] = t.getAttributes.collect({
      case (n, l) if l.isInstanceOf[ExecutionStateListener] => l.asInstanceOf[ExecutionStateListener]
    }).toSet
    def recursiveCollectListeners(t: TaskExecutionContext): Set[ExecutionStateListener] = collectListeners(t) ++ t.childContexts.values.flatMap(recursiveCollectListeners)
    recursiveCollectListeners(getRootContext(this))
  }

  private[tasker] def listeners: Set[ExecutionStateListener] = {
    val collect: Set[ExecutionStateListener] = attributes.collect({
      case (n, l) if l.isInstanceOf[ExecutionStateListener] => l.asInstanceOf[ExecutionStateListener]
    }).toSet
    parent match {
      case None => collect
      case Some(p) => collect ++ p.listeners
    }
  }

  def clearChildContexts() = childContexts.clear()

  def getAttribute(name: String): AnyRef = attributes.getOrElse(name, parent match {
    case None => null
    case Some(p) => p.getAttribute(name)
  }) match {
    case l: OldExecutionContextListenerCleanupTrigger => l.getWrappedListener
    case l@_ => l
  }

  def setAttribute(name: String, v: AnyRef) = v match {
    case ecl: ExecutionContextListener => attributes.put(name, new OldExecutionContextListenerCleanupTrigger(name, ecl))
    case x@_ => attributes.put(name, x)
  }

  private[tasker] def getAttributes = attributes

  def unsetAttribute(name: String) = attributes.remove(name)

  def getRepository = repository

  def stepContext(step: TaskStep, task: Task) = new ExecutionContext {
    val ERROR_PREFIX = "[ERROR]: "

    val stepLogger = Logger.apply(step.getImplementation.getClass)

    def setAttribute(name: String, value: scala.AnyRef) {
      TaskExecutionContext.this.setAttribute(name, value)
    }

    def logOutput(output: String) {
      stepLogger.info(output)
      step.logBuilder.append(output).append("\n")
      step.touch()
    }

    def logError(error: String) {
      stepLogger.error(error)
      appendErrorToLog(error)
      step.touch()
    }

    def logError(error: String, t: Throwable) {
      stepLogger.error(error, t)
      appendErrorToLog(error)
      val stringWriter: StringWriter = new StringWriter
      t.printStackTrace(new PrintWriter(stringWriter))
      for (stackTraceLine <- stringWriter.toString.split("\n")) {
        appendErrorToLog(stackTraceLine)
      }
      step.touch()
    }

    def appendErrorToLog(error: String) {
      step.logBuilder.append(ERROR_PREFIX).append(error).append("\n")
    }

    def getAttribute(name: String): AnyRef = attributes.getOrElse(name, null) match {
      case l: OldExecutionContextListenerCleanupTrigger => l.getWrappedListener
      case l@_ => l
    }

    def getInspectionContext: InspectionContext = spec.getInspectionContext

    def getRepository: Repository = repository

    def getTask: ITask = new ITask {

      override def getMetadata: util.Map[String, String] = task.getMetadata

      override def getUsername: String = task.getOwner

      override def getId: String = task.getId
    }

  }
}

object TaskExecutionContext {
  val CACHE_KEY = "taskExecutionContext"

}