package com.xebialabs.deployit.engine.tasker

import java.util.UUID

import akka.actor.{Actor, ActorRef, Props}
import com.xebialabs.deployit.engine.api.execution.StepExecutionState._
import com.xebialabs.deployit.engine.api.execution.{StepExecutionState, TaskState}
import com.xebialabs.deployit.engine.tasker.StateChangeEventListenerActor.StepStateEvent
import com.xebialabs.deployit.engine.tasker.StepExecutingActor.messages.{ExecuteStep, RetryStep}
import com.xebialabs.deployit.engine.tasker.messages.StepStateEventHandled
import com.xebialabs.deployit.plugin.api.flow.StepExitCode._
import com.xebialabs.deployit.plugin.api.flow.{ExecutionContext, StepExitCode}
import com.xebialabs.xlplatform.settings.CommonSettings
import grizzled.slf4j.Logging
import org.springframework.security.core.context.SecurityContextHolder

import scala.concurrent.duration.FiniteDuration

object StepExecutingActor {
  def props(task: TaskState, ctx: TaskExecutionContext) = Props(classOf[StepExecutingActor], task, ctx)

  object messages {
    case class ExecuteStep(taskStep: TaskStep, executionCtx: ExecutionContext)
    case class RetryStep(taskStep: TaskStep, executionContext: ExecutionContext, origSender: ActorRef)
  }
}

class StepExecutingActor(task: Task, ntec: TaskExecutionContext) extends Actor with Logging {
  val settings = CommonSettings(context.system).tasker

  def receive = ReceiveWithMdc(task) {
    case ExecuteStep(s, ctx) => executeStep(s, ctx, sender())
    case RetryStep(s, ctx, origSender) => executeStep(s, ctx, origSender, retry = true)
  }

  def executeStep(step: TaskStep, ctx: ExecutionContext, sender: ActorRef, retry: Boolean = false) {
    val implementation = step.getImplementation
    step.setRunner()
    if (!retry) {
      if (step.getState == StepExecutionState.EXECUTING) throw new IllegalStateException(s"The step [$step] is still executing, cannot run again.")
      if (step.getState.isFinal) {
        debug(s"Will not execute: $implementation with description: ${implementation.getDescription} because it has state: ${step.getState}")
        sender ! step.getState
        return
      }
      step.recordStart()

      if (step.isMarkedForSkip) {
        step.recordCompletion()
        setStepState(step, SKIPPED, sender)
        return
      }
      setStepState(step, StepExecutionState.EXECUTING, sender)
      step.clearLog()
      info(s"Started $implementation")
    } else {
      info(s"Continuing $implementation")
    }

    try {
      SecurityContextHolder.getContext.setAuthentication(task.getAuthentication)
      val result: StepExitCode = implementation.execute(ctx)
      step.recordCompletion()
      result match {
        case SUCCESS => setStepState(step, DONE, sender)
        case FAIL => setStepState(step, FAILED, sender)
        case PAUSE => setStepState(step, PAUSED, sender)
        case RETRY => {
          val delay: FiniteDuration = settings.stepRetryDelay
          ctx.logOutput(s"Retrying in ${delay.toSeconds} seconds...")
          context.system.scheduler.scheduleOnce(delay, self, RetryStep(step, ctx, sender))(context.dispatcher)
        }
      }
    }
    catch {
      case exc: Exception =>
        ctx.logError("Step failed", exc)
        step.recordCompletion()
        setStepState(step, FAILED, sender)
      case t: Throwable =>
        ctx.logError("Step failed badly, aborting!", t)
        step.recordCompletion()
        setStepState(step, FAILED, sender)
    }
    finally {
      SecurityContextHolder.getContext.setAuthentication(null)
      TaskStep.logger.info(step.getState.name)
    }
  }

  def setStepState(step: TaskStep, state: StepExecutionState, origSender: ActorRef) {
    import context._
    val oldState: StepExecutionState = step.getState
    step.setState(state)
    val taskId = task.getId
    val stepId: String = UUID.randomUUID().toString
    val waitForHandled: Boolean = state != EXECUTING
    if (waitForHandled) {
      context.system.eventStream.subscribe(this.self, classOf[StepStateEventHandled])
    }
    //    probably need a better identification of the event, ie. who is it for?
    context.system.eventStream.publish(StepStateEvent(task.getId, stepId, task, step, oldState, state, Some(ntec)))

    if (waitForHandled) {
      become(ReceiveWithMdc(task) {
        case StepStateEventHandled(`taskId`, `stepId`, `oldState`, `state`) =>
          debug(s"Handled state change [$oldState->$state]")
          become(receive)
          origSender ! state
      })
    }
  }
}



