package com.xebialabs.deployit.engine.tasker

import akka.actor.Status.Failure
import akka.actor._
import com.xebialabs.deployit.engine.api.execution.{BlockExecutionState, StepExecutionState}
import com.xebialabs.deployit.engine.tasker.BlockExecutingActor.{BlockDone, BlockStateChanged}
import com.xebialabs.deployit.engine.tasker.StepExecutingActor.messages.ExecuteStep
import com.xebialabs.deployit.engine.tasker.messages._

object StepBlockExecutingActor {

  def props(task: Task, block: StepBlock, ctx: TaskExecutionContext) =
    Props(classOf[StepBlockExecutingActor], task, block, ctx)

}

class StepBlockExecutingActor(task: Task, sb: StepBlock, ctx: TaskExecutionContext) extends BaseExecutionActor with ModifyStepsSupport {

  import com.xebialabs.deployit.engine.api.execution.BlockExecutionState._
  import context._

  var block: StepBlock = sb
  val taskId = task.getId

  private val blockFinder : BlockFinder = { blockPath =>
    if(block.getId() == blockPath.toBlockId) Option(block) else None
  }

  def receive: Actor.Receive = ReceiveWithMdc(task) (modifySteps(taskId, blockFinder) orElse start)

  private def start: Receive = {
    case Start(`taskId`) if Set(PENDING, STOPPED, ABORTED, FAILED).contains(block.state) =>
      updateStateAndNotify(BlockExecutionState.EXECUTING)
      executeStep(0, createChild(StepExecutingActor.props(task, ctx).withDispatcher("task.StepExecutorDispatcher")))
    case Start(`taskId`) if block.state == DONE =>
      debug(s"[${block.id}] : Not Executing again as it is already [EXECUTED]")
      changeState(block.state, None)
    case Start(`taskId`) => sender() ! Failure(new IllegalStateException(s"Can not execute a step block in state [${block.state}]"))
    case m@_ =>
      error(s"[${block.id}] : Don't know what to do with [$m]")
      sender() ! Failure(new IllegalStateException(s"$m"))
  }

  def executeStep(stepNr: Int, stepActor: ActorRef) {
    if (block.steps.size <= stepNr) {
      changeState(BlockExecutionState.DONE, Option(stepActor))
    } else {
      val head = block.steps(stepNr).asInstanceOf[TaskStep]
      become(waitForStep(head, stepNr + 1, stepActor))
      stepActor ! ExecuteStep(head, ctx.stepContext(head, task))
    }
  }

  def waitForStep(step: TaskStep, nextStepNr: Int, stepActor: ActorRef): Actor.Receive = ReceiveWithMdc(task) {
    case StepExecutionState.DONE | StepExecutionState.SKIPPED => executeStep(nextStepNr, stepActor)
    case StepExecutionState.PAUSED => changeState(BlockExecutionState.STOPPED, Option(stepActor))
    case StepExecutionState.FAILED => changeState(BlockExecutionState.FAILED, Option(stepActor))
    case Stop(`taskId`) =>
      info(s"[${block.id}] : Received [Stop] message")
      updateStateAndNotify(BlockExecutionState.STOPPING)
      become(stopAfterCurrentStep(step, nextStepNr, stepActor))
    case Abort(`taskId`) =>
      doAbort(stepActor, step)
  }


  def doAbort(stepActor: ActorRef, step: TaskStep) {
    info(s"[${block.id}] : Received [Abort] message")
    updateStateAndNotify(BlockExecutionState.ABORTING)
    context.watch(stepActor)
    become(aborting(step))
    step.interruptRunner()
    stepActor ! Kill
  }

  def aborting(step: TaskStep): Actor.Receive = ReceiveWithMdc(task) {
    case m@Terminated(stepActor) =>
      info(s"[${block.id}] : Received [$m], going to aborted state")
      step.setState(StepExecutionState.FAILED)
      changeState(BlockExecutionState.ABORTED, None)
    case m@_ =>
      info(s"[${block.id}] : Received [$m], ignoring")
  }

  def changeState(state: BlockExecutionState, stepActorOption: Option[ActorRef]) {
    state match {
      case DONE =>
        doNotAcceptMessages("Block done")
      case ABORTED | FAILED | STOPPED =>
        stepActorOption.foreach(context.stop)
        become(modifySteps(taskId, blockFinder) orElse receive)
      case _ =>
        stepActorOption.foreach(context.stop)
        become(start)
    }
    updateStateAndNotify(state)
    parent ! BlockDone(task.getId, block)
  }


  def stopAfterCurrentStep(step: TaskStep, nextStepNr: Int, stepActor: ActorRef): Actor.Receive = ReceiveWithMdc(task) {
    case Abort(`taskId`) =>
      doAbort(stepActor, step)
    case StepExecutionState.DONE if block.steps.size <= nextStepNr => changeState(BlockExecutionState.DONE, Option(stepActor))
    case m@_ => changeState(BlockExecutionState.STOPPED, Option(stepActor))
  }

  def updateStateAndNotify(newState: BlockExecutionState) {
    val oldState: BlockExecutionState = block.state
    if (oldState != newState) {
      block = block.newState(newState)
      val msg: BlockStateChanged = BlockStateChanged(task.getId, block, oldState, newState)
      debug(s"[${block.id}] : Sending BlockStateChange($oldState->$newState) message for [${block.id}]")
      parent ! msg
    }
  }
}
