package com.xebialabs.deployit.engine.tasker

import java.util

import com.xebialabs.deployit.engine.api.execution._
import com.xebialabs.deployit.engine.tasker.step.PauseStep
import com.xebialabs.deployit.plugin.api.flow.Step

import scala.collection.mutable.ListBuffer
import scala.language.implicitConversions

sealed trait Block extends BlockState {
  def id: BlockPath

  def description: Description

  private[tasker] var state: BlockExecutionState = BlockExecutionState.PENDING

  def getStepList(): java.util.List[StepState]

  def getDescription() = description

  def getId() = id.toBlockId

  def getState() = state

  def addPause(position: BlockPath): Unit

  def getStep(path: BlockPath): StepState

  def getBlock(path: BlockPath): Block

  private[tasker] def getStepsWithPaths(): Seq[(BlockPath, StepState)] = this match {
    case x: CompositeBlock => x.blocks.flatMap(b => b.getStepsWithPaths())
    case x: StepBlock => x.steps.zipWithIndex.map({ case (s, i) => (x.id / (i + 1), s)})
  }


  def recovered(): Block = {
    import com.xebialabs.deployit.engine.api.execution.BlockExecutionState._
    if (Set(PENDING, DONE, FAILED, STOPPED, ABORTED).contains(state)) this
    else {
      val recoveredState: BlockExecutionState = state match {
        case EXECUTING | STOPPING | FAILING => FAILED
        case ABORTING => ABORTED
        case _ => state
      }
      this.state = recoveredState
      this match {
        case x: CompositeBlock => x.blocks.foreach(_.recovered())
        case x: StepBlock => x.steps.foreach(_.asInstanceOf[TaskStep].recovered())
      }
      this
    }
  }

  // Butt UGLY!
  def newState[A <: Block](state: BlockExecutionState): A = {
    this.state = state
    this.asInstanceOf[A]
  }
}

sealed trait ImmutableBlock extends Block {

  import scala.collection.JavaConversions._

  def getStepList(): java.util.List[StepState] = getStepsWithPaths().map(_._2)
}


sealed trait CompositeBlock extends ImmutableBlock with CompositeBlockState {

  import scala.collection.convert.wrapAll._

  def blocks: Seq[Block]

  def determineNewState(block: Block, remainingBlocks: Seq[Block]): BlockExecutionState

  def getBlocks() = blocks.toList

  def getBlock(position: BlockPath) = position match {
    case p if p.isEmpty => this
    case p if p.head > blocks.size => throw new IllegalArgumentException(s"Cannot add a pause step to a non-existing block [${p.head}] of composite block [$getId]")
    case p => blocks.get(p.head - 1).getBlock(p.tail)
  }

  def addPause(position: BlockPath) = position match {
    case _ if state == BlockExecutionState.DONE => throw new IllegalArgumentException(s"Cannot add a pause step to an block that is done [$getId]")
    case p if p.isEmpty => throw new IllegalArgumentException(s"Cannot add a pause step to composite block [$getId]")
    case p => blocks.get(p.head - 1).addPause(p.tail)
  }

  def getStep(position: BlockPath): StepState = position match {
    case p if p.isEmpty => throw new IllegalArgumentException(s"Cannot get step at path [$getId]")
    case p if p.head > blocks.size => throw new IllegalArgumentException(s"Cannot get step [$p] at block [$getId]")
    case p => blocks.get(p.head - 1).getStep(p.tail)
  }
}

case class ParallelBlock(id: BlockPath, description: Description, blocks: Seq[Block]) extends CompositeBlock {

  def determineNewState(block: Block, remainingBlocks: Seq[Block]): BlockExecutionState = {
    import com.xebialabs.deployit.engine.api.execution.BlockExecutionState._
    (state, block.state) match {
      case (PENDING | DONE | FAILED | STOPPED | ABORTED, _) => throw new IllegalStateException(s"Cannot receive state updates when $state")
      case (ABORTING, DONE | FAILED | STOPPED) | (_, ABORTED) => if (remainingBlocks.isEmpty) ABORTED else ABORTING
      case (ABORTING, _) | (_, ABORTING) => ABORTING
      case (FAILING, STOPPED | DONE) | (_, FAILED) => if (remainingBlocks.isEmpty) FAILED else FAILING
      case (FAILING, _) | (_, FAILING) => FAILING
      case (STOPPING, DONE) if remainingBlocks.isEmpty && allDone => DONE
      case (STOPPING, DONE) | (_, STOPPED) => if (remainingBlocks.isEmpty) STOPPED else STOPPING
      case (_, STOPPING) => block.state
      case (EXECUTING, DONE) => if (remainingBlocks.isEmpty) DONE else EXECUTING
      case (_, EXECUTING) => state
    }
  }

  def allDone() = blocks.filterNot(_.state == BlockExecutionState.DONE).isEmpty

  def isParallel() = true
}

case class SerialBlock(id: BlockPath, description: Description, blocks: Seq[Block]) extends CompositeBlock {

  def determineNewState(block: Block, remainingBlocks: Seq[Block]): BlockExecutionState = {
    import com.xebialabs.deployit.engine.api.execution.BlockExecutionState._
    (state, block.state) match {
      case (_, DONE) => if (remainingBlocks.isEmpty) DONE else state
      case _ => block.state
    }
  }

  def isParallel() = false

}

case class StepBlock(id: BlockPath, description: Description, steps: ListBuffer[StepState]) extends Block with StepBlockState {

  import scala.collection.JavaConversions._

  def getStepList(): util.List[StepState] = steps

  def getSteps: util.List[StepState] = steps

  def getCurrentStep: Int = steps.indexWhere(_.getState == StepExecutionState.EXECUTING) + 1

  def addPause(position: BlockPath) = position match {
    case _ if state == BlockExecutionState.DONE => throw new IllegalArgumentException(s"Cannot add a pause step to an block that is done [$getId]")
    case p if p.isEmpty => throw new IllegalArgumentException(s"No position in step block [$getId] given for adding pause step")
    case p if (p.head - 1) > steps.size => throw new IllegalArgumentException(s"Wrong position [${p.head}] in step block [$getId] given for adding pause step")
    case p if (p.head - 1) != steps.size && !List(StepExecutionState.PENDING, StepExecutionState.SKIP).contains(steps(p.head - 1).getState) => throw new IllegalArgumentException(s"Add pause step before step [${p.head}] at block [$getId]: state must be PENDING or SKIP, was [${steps(p.head - 1).getState}]")
    case p => steps.insert(p.head - 1, new TaskStep(new PauseStep))
  }

  def getStep(position: BlockPath): StepState = position match {
    case p if p.isEmpty => throw new IllegalArgumentException(s"No position in step block [$getId] given")
    case p if (p.head - 1) > steps.size => throw new IllegalArgumentException(s"Wrong position [${p.head}] in step block [$getId] given")
    case p => steps.get(p.head - 1)
  }

  override def getBlock(path: BlockPath): Block = if (path.isEmpty) this else throw new IllegalArgumentException(s"Cannot get Block [${path.head}] at block [$getId]")
}

sealed trait BlockBuilder {
  def build(): Block = build(BlockPath(0))

  def build(id: BlockPath): Block
}

trait CompositeBlockBuilder extends BlockBuilder

case class StepBlockBuilder(description: Description, steps: ListBuffer[StepState]) extends BlockBuilder {
  def build(id: BlockPath): Block = StepBlock(id, description, steps)
}

case class ParallelBlockBuilder(description: Description, blockBuilders: Seq[BlockBuilder]) extends CompositeBlockBuilder {
  def build(id: BlockPath) = ParallelBlock(id, description, blockBuilders.zipWithIndex.map { case (bb, i) => bb.build(id.newSubPath(i + 1))})
}

case class SerialBlockBuilder(description: Description, blockBuilders: Seq[BlockBuilder]) extends CompositeBlockBuilder {
  def build(id: BlockPath) = SerialBlock(id, description, blockBuilders.zipWithIndex.map { case (bb, i) => bb.build(id.newSubPath(i + 1))})
}

object BlockBuilders {

  import scala.collection.convert.wrapAll._

  def parallel(description: Description, blockBuilders: BlockBuilder*) = ParallelBlockBuilder(description, blockBuilders)

  def parallel(description: Description, blockBuilders: List[BlockBuilder]) = ParallelBlockBuilder(description, blockBuilders)

  def parallel(description: Description, blockBuilders: util.List[BlockBuilder]) = ParallelBlockBuilder(description, blockBuilders)

  def serial(description: Description, blockBuilders: BlockBuilder*) = SerialBlockBuilder(description, blockBuilders)

  def serial(description: Description, blockBuilders: List[BlockBuilder]) = SerialBlockBuilder(description, blockBuilders)

  def serial(description: Description, blockBuilders: util.List[BlockBuilder]) = SerialBlockBuilder(description, blockBuilders)

  def steps(description: Description, steps: List[StepState]) = StepBlockBuilder(description, ListBuffer.apply(steps: _*))

  def steps(description: Description, steps: util.List[StepState]) = StepBlockBuilder(description, ListBuffer.apply(steps: _*))

  // auto-convert from Step -> StepState, different name because of generic erasure
  def step(description: Description, steps: List[Step]) = StepBlockBuilder(description, ListBuffer.apply(steps.map(new TaskStep(_)): _*))

  def step(description: Description, steps: util.List[Step]) = StepBlockBuilder(description, ListBuffer.apply(steps.map(new TaskStep(_)): _*))

  implicit def blockBuilderToBlock(bb: BlockBuilder): Block = bb.build()

  implicit def blockBuilderToBlock(bb: CompositeBlockBuilder): CompositeBlock = bb.build().asInstanceOf[CompositeBlock]
}

case class BlockPath(elems: List[Int]) {
  def toBlockId: BlockId = elems.mkString("_")

  def newSubPath(elem: Int): BlockPath = new BlockPath(elems ::: List(elem))

  def /(elem: Int) = newSubPath(elem)

  def tail = new BlockPath(elems.tail)

  def init = new BlockPath(elems.init)

  def head = elems.head

  def isEmpty = elems.isEmpty

  def isLeaf = elems.size == 1
}

object BlockPath {
  def apply(pathString: String) = new BlockPath(pathString.split('_').toList.map(_.toInt))

  def apply(elem: Int) = new BlockPath(List(elem))
}
