package com.xebialabs.deployit.engine.tasker

import java.io._
import java.nio.file.{Files, StandardCopyOption}
import javassist.util.proxy.{ProxyObjectInputStream, ProxyObjectOutputStream}
import javax.crypto.SealedObject

import akka.actor.{Actor, Cancellable, Props}
import com.xebialabs.deployit.engine.tasker.StateChangeEventListenerActor.{StepStateEvent, TaskStateEvent}
import com.xebialabs.deployit.engine.tasker.TaskRecoveryActor.Write
import com.xebialabs.deployit.engine.tasker.messages.TaskStateEventHandled
import com.xebialabs.deployit.util.SecretKeyHolder
import grizzled.slf4j.Logging
import org.slf4j.LoggerFactory

import scala.concurrent.duration._
import scala.language.postfixOps
import scala.reflect.ClassTag
import scala.util.{Failure, Success, Try}

object TaskRecoveryActor {
  def props(taskId: TaskId, recoveryDir: File) = Props(classOf[TaskRecoveryActor], taskId, recoveryDir)

  case class Write(task: Task)
}

class TaskRecoveryActor(taskId: TaskId, recoveryDir: File) extends Actor with Logging with RecoveryWriter {

  import context._

  if (!recoveryDir.exists()) recoveryDir.mkdirs()

  implicit private val implRecoveryDir = recoveryDir

  def receive: Actor.Receive = ReceiveWithMdc(taskId) {
    case e @ TaskStateEvent(`taskId`, _, _, state) if !state.isFinal =>
      debug(s"Received [$e]")
      become(writeScheduled(system.scheduler.scheduleOnce(1 second, self, Write(e.t))))
      debug(s"Scheduled Recovery writer for [$taskId] for 1 second from now")
    case e @ StepStateEvent(`taskId`, _, _, _, _, _, _) =>
      debug(s"Received [$e]")
      become(writeScheduled(system.scheduler.scheduleOnce(1 second, self, Write(e.t))))
      debug(s"Scheduled Recovery writer for [$taskId] for 1 second from now")
    case e @ TaskStateEvent(`taskId`, _, oldState, state) if state.isFinal =>
      debug(s"Cleaning up recovery file for [$taskId]")
      deleteRecoveryFile(taskId)
      system.eventStream.publish(TaskStateEventHandled(taskId, oldState, state))
  }

  private def writeScheduled(token: Cancellable): Actor.Receive = ReceiveWithMdc(taskId) {
    case e @ TaskStateEvent(`taskId`, _, oldState, state) if state.isFinal =>
      token.cancel()
      debug(s"Canceled writing recovery file for [$taskId]")
      deleteRecoveryFile(taskId)
      context.system.eventStream.publish(TaskStateEventHandled(taskId, oldState, state))
    case Write(task) =>
      debug(s"Writing recovery file for: [$task]")
      writeSealedTask(task, SecretKeyHolder.get())
      become(receive)
    case e @ StepStateEvent =>
  }
}

trait RecoveryWriter {

  private[this] val log = LoggerFactory.getLogger(getClass)

  def writeSealedTaskTo(file: File, task: Task, keyHolder: SecretKeyHolder): File = {
    writeToFile(file, new SealedObject(taskToByteArray(task), keyHolder.getEncryption))
  }

  private def taskToByteArray(task: Task): TaskByteArrayWrapper = {
    import com.xebialabs.xlplatform.utils.ResourceManagement._
    val taskByteStream = new ByteArrayOutputStream()
    using(managedClosable(new ProxyObjectOutputStream(taskByteStream))) { os =>
      os.writeObject(task)
    }
    new TaskByteArrayWrapper(taskByteStream.toByteArray)
  }

  def writeSealedTask(task: Task, keyHolder: SecretKeyHolder)(implicit recoveryDir: File): File = {
    writeSealedTaskTo(recoveryFile(task.getId), task, keyHolder)
  }

  def writeToFile[T](file: File, any: T) = {
    val tmpFile = new File(s"${file.getAbsolutePath}.tmp")
    import com.xebialabs.xlplatform.utils.ResourceManagement._
    try {
      using(managedClosable(new ProxyObjectOutputStream(new FileOutputStream(tmpFile)))) { os =>
        os.writeObject(any)
      }
      Files.move(tmpFile.toPath, file.toPath, StandardCopyOption.REPLACE_EXISTING, StandardCopyOption.ATOMIC_MOVE)
      file
    } catch {
      case e: IOException =>
        log.error(s"Could not write recovery file [$file]", e)
        throw e
    }
  }

  def deleteRecoveryFile(id: String)(implicit recoveryDir: File) = recoveryFile(id).delete()

  def recoveryFile(id: String)(implicit recoveryDir: File): File = new File(recoveryDir, s"$id.task")
}

trait TaskRecovery extends RecoveryWriter {
  private[this] val log = LoggerFactory.getLogger(getClass)

  def recover(file: File)(implicit keyHolder: SecretKeyHolder): Option[Task] = {
    securedRecover(file).recoverWith {
      case ex: Exception =>
        log.debug("Secured recovery failed. Trying plain recovery", ex)
        plainRecover(file).map {
          case task: Task =>
            convertPlainToSealed(file, task)
            task
        }
    } match {
      case Success(task) =>
        task.recovered()
        if (task.getState.isFinal) {
          log.info(s"Task [${task.getId}] was recovered in final state [${task.getState}]. Removing recovery file.")
          file.delete()
          None
        } else {
          Option(task)
        }
      case Failure(e) if e.isInstanceOf[ClassNotFoundException] =>
        log.error(s"Could not find serialized class in recovery file [$file]", e)
        None
      case Failure(e) =>
        log.error(s"Could not read recovery file [$file]", e)
        None
    }
  }

  def convertPlainToSealed(file: File, task: Task)(implicit keyHolder: SecretKeyHolder): Task = {
    log.info(s"Converting plain task [$file] to sealed task")
    writeSealedTaskTo(file, task, keyHolder)
    task
  }

  def byteArrayToTask(wrapper: TaskByteArrayWrapper): Try[Task] = {
    import com.xebialabs.xlplatform.utils.ResourceManagement._
    Try(using(managedClosable(new CLProxyObjectInputStream(new ByteArrayInputStream(wrapper.getSerializedTask)))) { is =>
      is.readObject.asInstanceOf[Task]
    })
  }

  def securedRecover(file: File)(implicit keyHolder: SecretKeyHolder): Try[Task] = {
    val buffer: Try[TaskByteArrayWrapper] = Try {
      recover[SealedObject](file).map(_.getObject(keyHolder.getDecryption)).map(_.asInstanceOf[TaskByteArrayWrapper])
    }.flatten
    buffer.flatMap(byteArrayToTask)
  }

  def plainRecover(file: File) = recover[Task](file)

  private def recover[T: ClassTag](file: File): Try[T] = {
    import com.xebialabs.xlplatform.utils.ResourceManagement._
    log.info(s"Recovering [${implicitly[ClassTag[T]].runtimeClass.getName}] from file [$file]")
    Try(using(managedClosable(new CLProxyObjectInputStream(new FileInputStream(file)))) { is =>
      is.readObject.asInstanceOf[T]
    })
  }
}

class CLProxyObjectInputStream(input: InputStream) extends ProxyObjectInputStream(input) {
  override def resolveClass(desc: ObjectStreamClass): Class[_] = {
    Try(super.resolveClass(desc)).getOrElse(Class.forName(desc.getName))
  }
}

