package com.xebialabs.deployit.engine.tasker

import akka.actor.{ActorRef, Props}
import com.xebialabs.deployit.engine.tasker.satellite.SatellitePath

sealed trait SatelliteBlockRouter {

  def route(task: Task, block: CompositeBlock, ctx: TaskExecutionContext)(implicit context: akka.actor.ActorContext): Props

  def route(task: Task, block: StepBlock, ctx: TaskExecutionContext)(implicit context: akka.actor.ActorContext): Props

}

object SatelliteBlockRouter {

  object Master extends SatelliteBlockRouter {

    def route(task: Task, block: CompositeBlock, ctx: TaskExecutionContext)(implicit context: akka.actor.ActorContext): Props = routeToSatellite(task, block, ctx) getOrElse {
      BlockExecutingActor.props(task, block, ctx, Master)
    }

    def route(task: Task, block: StepBlock, ctx: TaskExecutionContext)(implicit context: akka.actor.ActorContext): Props = routeToSatellite(task, block, ctx) getOrElse {
      StepBlockExecutingActor.props(task, block, ctx)
    }

    private def routeToSatellite(task: Task, block: ExecutableBlock, ctx: TaskExecutionContext)(implicit context: akka.actor.ActorContext): Option[Props] = {
      block.satellite.map(sat => {
        BlockOnSatellite.props(task.getId, block, SatellitePath(sat), context.self)
      })
    }
  }

  object Satellite extends SatelliteBlockRouter {

    def route(task: Task, block: CompositeBlock, ctx: TaskExecutionContext)(implicit context: akka.actor.ActorContext): Props = block match {
      case _: StepBlock =>
        throw new IllegalArgumentException("I cannot handle Step Blocks...")

      case _ =>
        BlockExecutingActor.props(task, block, ctx, Master)
    }


    def route(task: Task, block: StepBlock, ctx: TaskExecutionContext)(implicit context: akka.actor.ActorContext): Props =
      StepBlockExecutingActor.props(task, block, ctx)
  }

}