package com.xebialabs.deployit.engine.tasker

import java.io.Serializable

import com.xebialabs.deployit.engine.spi.execution.ExecutionStateListener
import com.xebialabs.deployit.engine.tasker.log.StepLogFactory
import com.xebialabs.deployit.plugin.api.execution.ExecutionContextListener
import com.xebialabs.deployit.plugin.api.services.Repository

class TaskExecutionContext(parent: Option[TaskExecutionContext], spec: TaskSpecification, @transient var repository: Repository, @transient var stepLogFactory: StepLogFactory) extends Serializable {
  private[this] val attributes: Cache[String, AnyRef] = Cache()

  attributes += (TaskExecutionContext.CACHE_KEY -> this.spec)

  if (parent.isEmpty && spec != null) {
    spec.getListeners.forEach(l => attributes.put(l.getClass.getName, l))
  }

  private def getParent: Option[TaskExecutionContext] = parent

  val childContexts: Cache[BlockPath, TaskExecutionContext] = Cache()

  def contextFor(block: Block): TaskExecutionContext = contextFor(block.id)

  def contextFor(path: BlockPath): TaskExecutionContext = childContexts.getOrElse(path, {
    val childContext = new TaskExecutionContext(Some(this), spec, repository, stepLogFactory)
    childContexts.put(path, childContext)
    childContext
  })

  private[TaskExecutionContext] def listeners = attributes.collect {
    case (_, l: ExecutionStateListener) => l
  }.toSet

  private[tasker] def allListeners: Set[ExecutionStateListener] = {
    def getRootContext(tec: TaskExecutionContext): TaskExecutionContext = tec.getParent match {
      case Some(pp) => getRootContext(pp)
      case None => tec
    }

    def recursiveCollectListeners(t: TaskExecutionContext): Set[ExecutionStateListener] = t.listeners ++ t.childContexts.values.flatMap(recursiveCollectListeners)

    recursiveCollectListeners(getRootContext(this))
  }

  private[tasker] def inheritedListeners: Set[ExecutionStateListener] = parent match {
    case None => listeners
    case Some(p) => listeners ++ p.inheritedListeners
  }

  def clearChildContexts(): Unit = childContexts.clear()

  def getAttribute(name: String): AnyRef = attributes.getOrElse(name, parent.map(_.getAttribute(name)).orNull) match {
    case l: OldExecutionContextListenerCleanupTrigger => l.getWrappedListener
    case l@_ => l
  }

  def setAttribute(name: String, v: AnyRef): Option[AnyRef] = v match {
    case ecl: ExecutionContextListener => attributes.put(name, new OldExecutionContextListenerCleanupTrigger(name, ecl))
    case x@_ => attributes.put(name, x)
  }

  private[tasker] def getAttributes = attributes

  def unsetAttribute(name: String): Option[AnyRef] = attributes.remove(name)

  def getRepository: Repository = repository

  def stepContext(step: TaskStep, stepPath: BlockPath, task: Task) =
    new StepExecutionContext(this, step, task, spec, stepLogFactory.create(task.getId, stepPath, step))

}

object TaskExecutionContext {
  val CACHE_KEY = "taskExecutionContext"

}