package com.xebialabs.deployit.tasksystem

import akka.actor.ActorSystem
import com.typesafe.config.ConfigFactory._
import com.typesafe.config.{Config, ConfigFactory}
import com.xebialabs.deployit.security.SecretKeyHolder
import com.xebialabs.xlplatform.config._
import com.xebialabs.xlplatform.settings.{CommonSettings, ConfigSupport, SecuritySettings}

object TaskActorSystem extends ConfigSupport {

  lazy val actorSystem: ActorSystem = newActorSystem(overrideConfig.withFallback(parseResources("xl-deploy.conf")))

  var overrideConfig: Config = ConfigFactory.empty()

  def setOverrideConfig(newConfig: Config): Unit = {
    overrideConfig = newConfig
  }

  def newActorSystem(systemConfig: Config, systemName: Option[String] = None): ActorSystem = {
    val commonSettings = new CommonSettings(systemConfig.withFallback(ConfigFactory.defaultReference()))
    val actorConfig = if (systemConfig.hasPath("xl.task.system"))
      systemConfig.getConfig("xl.task.system").withFallback(systemConfig)
    else
      systemConfig
    ActorSystem.create(systemName.getOrElse("task-sys"), loadConfig(actorConfig, commonSettings))
  }

  def loadConfig(systemConfig: Config): Config = {
    loadConfig(systemConfig, new CommonSettings(systemConfig.withFallback(ConfigFactory.defaultReference())))
  }

  private def loadConfig(systemConfig: Config, commonSettings: CommonSettings): Config = {
    val satelliteEnabled = commonSettings.satellite.enabled
    val remotingEnabled = satelliteEnabled || !commonSettings.inProcessTaskEngine
    val remotingDisabled = !remotingEnabled

    val conditionalFallbacks: Seq[(Config, Boolean)] = Seq(
      hostAddressConfiguration("xl.server.hostname") -> remotingEnabled,
      sslRemoteTransports(commonSettings.security) -> remotingEnabled,
      parseResources("remote-enabled.conf") -> remotingEnabled,
      ConfigLoader.loadWithDynamic("xl-deploy.conf") -> remotingDisabled,
      parseResources("satellite-enabled.conf") -> satelliteEnabled,
    )
    loadWithFallbacks(systemConfig, conditionalFallbacks.filter(_._2).map(_._1)  : _*).withFallback(ConfigFactory.defaultReference()).decrypted(SecretKeyHolder.get())
  }

  private[this] def sslRemoteTransports(security: SecuritySettings): Config = {
    import scala.collection.JavaConverters._
    import scala.collection.convert.ImplicitConversionsToJava._
    parseMap(Map("akka.remote.enabled-transports" -> Seq(if (security.enabled) "akka.remote.netty.ssl" else "akka.remote.netty.tcp").asJava))
  }
}
