package com.xebialabs.xlplatform.settings

import java.io.File

import akka.actor.{ExtendedActorSystem, Extension, ExtensionId, ExtensionIdProvider}
import akka.util.Timeout
import com.typesafe.config.Config

import scala.concurrent.duration._
import scala.language.postfixOps

object CommonSettings extends ExtensionId[CommonSettings] with ExtensionIdProvider {

  override def createExtension(system: ExtendedActorSystem): CommonSettings = new CommonSettings(system.settings.config)

  override def lookup(): ExtensionId[_ <: Extension] = CommonSettings
}

class CommonSettings(config: Config) extends Extension {
  lazy val tasker = new TaskerSettings(config)
  lazy val satellite = new SatelliteSettings(config.getConfig("xl"))
  lazy val server = new ServerSettings(config.getConfig("xl"))
  lazy val worker = new WorkerSettings(config.getConfig("xl.worker"))
  lazy val scheduler = new SchedulerSettings(config) // get rid
  lazy val security = new SecuritySettings(config.getConfig("xl.server.ssl"))
  lazy val inProcessTaskEngine: Boolean = config.getBoolean("xl.task.in-process-worker")
}

class TaskerSettings(val config: Config) extends SettingsSupport {
  import SettingsSupport._
  lazy val recoveryDir = new File(value[String]("xl.task.recovery-dir"))
  lazy val maxActiveTasks: Int = value[Int]("xl.task.max-active-tasks")
  lazy val shutdownTimeout: FiniteDuration = value[FiniteDuration]("xl.task.shutdown-timeout")
  lazy val tickDuration: FiniteDuration = value[FiniteDuration]("akka.scheduler.tick-duration")
  lazy val stepRetryDelay: FiniteDuration = value[FiniteDuration]("xl.task.step.retry-delay")
  lazy val askTimeout: Timeout = valueWithDefault("xl.tasker.askTimeout", 10 seconds)
  lazy val stepRunTimeout: FiniteDuration = valueWithDefault("xl.task.step.run-timeout", 6 hours)
}

class SatelliteSettings(val config: Config) extends SettingsSupport {
  import SettingsSupport._
  lazy val uploadIdleTimeout: Timeout = valueWithDefault("satellite.timeout.upload.idle", 10 seconds)
  lazy val infoTimeout: Timeout = valueWithDefault("satellite.timeout.info", 10 seconds)
  lazy val pingTimeout: Timeout = valueWithDefault("satellite.timeout.ping", 5 seconds)
  lazy val remoteAskTimeout: Timeout = valueWithDefault("satellite.timeout.remoteAsk", 60 seconds)
  lazy val enabled: Boolean = valueWithDefault("satellite.enabled", false)
  lazy val streamingConnectionTimeout: Timeout = valueWithDefault("satellite.timeout.streaming", 10 seconds)
  lazy val maxConcurrentUploads: Int = valueWithDefault("satellite.streaming.max-uploads", 10)
  lazy val fileUploadAttempts: Int = valueWithDefault("satellite.streaming.upload-attempts", 3)
}

class ServerSettings(val config: Config) extends SettingsSupport {
  import SettingsSupport._
  lazy val hostname: String = value[String]("server.hostname")
  lazy val aggregationTimeout: FiniteDuration = valueWithDefault("server.aggregation-timeout", 9 seconds)
}

class WorkerSettings(val config: Config) extends SettingsSupport {
  import SettingsSupport._
  lazy val connectInterval: FiniteDuration = valueWithDefault("connect.interval", 10 seconds)
  lazy val reconnectDelay: FiniteDuration = valueWithDefault("reconnect.delay", 30 seconds)
  lazy val reconnectInterval: FiniteDuration = valueWithDefault("reconnect.interval", 10 seconds)
}

// get rid
class SchedulerSettings(val config: Config) extends SettingsSupport {
  import SettingsSupport._
  lazy val tickDuration: FiniteDuration = value[FiniteDuration]("akka.scheduler.tick-duration")
}