package com.xebialabs.xlplatform.settings

import java.io.File
import java.util.concurrent.TimeUnit
import java.util

import akka.actor.{ExtendedActorSystem, Extension, ExtensionId, ExtensionIdProvider}
import akka.util.Timeout
import com.typesafe.config.Config

import scala.concurrent.duration._

object CommonSettings extends ExtensionId[CommonSettings] with ExtensionIdProvider {

  override def createExtension(system: ExtendedActorSystem): CommonSettings = new CommonSettings(system.settings.config)

  override def lookup(): ExtensionId[_ <: Extension] = CommonSettings
}

class CommonSettings(config: Config) extends Extension {
  lazy val tasker = new TaskerSettings(config)
  lazy val satellite = new SatelliteSettings(config)
  lazy val scheduler = new SchedulerSettings(config)
  lazy val security = new SecuritySettings(config.getConfig("satellite.ssl"))
}

class TaskerSettings(val config: Config) extends SettingsSupport {
  lazy val recoveryDir = new File(value[String]("task.recovery-dir"))
  lazy val tickDuration = value[FiniteDuration]("akka.scheduler.tick-duration")
  lazy val stepRetryDelay = value[FiniteDuration]("task.step.retry-delay")
  lazy val askTimeout: Timeout = valueWithDefault("tasker.askTimeout", FiniteDuration(1, TimeUnit.SECONDS))
}

class SatelliteSettings(val config: Config) extends SettingsSupport {
  lazy val uploadIdleTimeout: Timeout = valueWithDefault("satellite.timeout.upload.idle", FiniteDuration(10, TimeUnit.SECONDS))
  lazy val pingTimeout: Timeout = valueWithDefault("satellite.timeout.ping", FiniteDuration(5, TimeUnit.SECONDS))
  lazy val remoteAskTimeout: Timeout = valueWithDefault("satellite.timeout.remoteAsk", FiniteDuration(60, TimeUnit.SECONDS))
  lazy val hostname = value[String]("satellite.hostname")
  lazy val enabled = valueWithDefault("satellite.enabled", false)
  lazy val streamingConnectionTimeout: Timeout = valueWithDefault("satellite.timeout.streaming", FiniteDuration(10, TimeUnit.SECONDS))
  lazy val maxConcurrentUploads: Int = valueWithDefault("satellite.streaming.max-uploads", 10)
}

class SecuritySettings(val config: Config) extends SettingsSupport {

  import collection.convert.wrapAsScala._

  lazy val enabled = valueWithDefault[Boolean]("enabled", false)

  lazy val enabledAlgorithms: Seq[String] = value[util.List[String]]("enabled-algorithms")

  lazy val trustStore = value[String]("trust-store")
  lazy val trustStorePassword = value[String]("trust-store-password")

  lazy val keyStore = valueOption[String]("key-store")
  lazy val keyPassword = value[String]("key-password")
  lazy val keyStorePassword = value[String]("key-store-password")

  lazy val protocol = value[String]("protocol")
}

class SchedulerSettings(val config: Config) extends SettingsSupport {
  lazy val tickDuration = value[FiniteDuration]("akka.scheduler.tick-duration")
}

trait SettingsSupport {

  type ValueExtractor[T] = (String) => T
  val config: Config

  implicit val stringExtractor: ValueExtractor[String] = config.getString
  implicit val intExtractor: ValueExtractor[Int] = config.getInt
  implicit val jListExtractor: ValueExtractor[util.List[String]] = config.getStringList
  implicit val booleanExtractor: ValueExtractor[Boolean] = config.getBoolean
  implicit val durationExtractor: ValueExtractor[FiniteDuration] = (path: String) => FiniteDuration(config.getDuration(path, TimeUnit.MILLISECONDS), TimeUnit.MILLISECONDS)
  implicit val $conforms: ValueExtractor[SettingsSupport] = null // to shade Predef.conforms implicit

  def value[T](path: String)(implicit extractor: ValueExtractor[T]): T = extractor.apply(path)

  def valueOption[T](path: String)(implicit extractor: ValueExtractor[T]): Option[T] = if (config.hasPath(path)) Option(value(path)(extractor)) else None

  def valueWithDefault[T](path: String, default: T)(implicit extractor: ValueExtractor[T]) = valueOption(path)(extractor).getOrElse(default)
}