package com.xebialabs.xlplatform.settings

import java.io.File
import java.util.concurrent.TimeUnit

import akka.actor.{ExtendedActorSystem, Extension, ExtensionId, ExtensionIdProvider}
import akka.util.Timeout
import com.typesafe.config.Config

import scala.concurrent.duration._

object CommonSettings extends ExtensionId[CommonSettings] with ExtensionIdProvider {

  override def createExtension(system: ExtendedActorSystem): CommonSettings = new CommonSettings(system.settings.config)

  override def lookup(): ExtensionId[_ <: Extension] = CommonSettings
}

class CommonSettings(config: Config) extends Extension {
  lazy val tasker = new TaskerSettings(config)
  lazy val satellite = new SatelliteSettings(config)
  lazy val scheduler = new SchedulerSettings(config)
}

class TaskerSettings(val config: Config) extends DefaultConfigurations {
  lazy val recoveryDir = new File(config.getString("task.recovery-dir"))
  lazy val tickDuration = config.getDuration("akka.scheduler.tick-duration", TimeUnit.MILLISECONDS)
  lazy val stepRetryDelay = FiniteDuration(config.getDuration("task.step.retry-delay", MILLISECONDS), MILLISECONDS)
  lazy val askTimeout = timeout("tasker.askTimeout", 1 * 1000, TimeUnit.MILLISECONDS)
}

class SatelliteSettings(val config: Config) extends DefaultConfigurations {
  lazy val uploadIdleTimeout: Timeout = timeout("satellite.timeout.upload.idle", 10 * 1000, TimeUnit.MILLISECONDS)
  lazy val pingTimeout = timeout("satellite.timeout.ping", 5 * 1000, TimeUnit.MILLISECONDS)
  lazy val remoteAskTimeout = timeout("satellite.timeout.remoteAsk", 60 * 1000, TimeUnit.MILLISECONDS)
  lazy val hostname = config.getString("satellite.hostname")
  lazy val enabled = valueWithDefault("satellite.enabled", false)(config.getBoolean)
  lazy val sslEnabled = valueWithDefault("satellite.ssl.enabled", false)(config.getBoolean)
}

class SchedulerSettings(val config: Config) extends DefaultConfigurations {
  lazy val tickDuration = config.getDuration("akka.scheduler.tick-duration", TimeUnit.MILLISECONDS)
}

trait DefaultConfigurations extends OptionConfigurations {
  val config: Config

  def timeout(configPath: String, defaultValue: Long, unit: TimeUnit) = {
    Timeout(FiniteDuration(if (config.hasPath(configPath)) config.getDuration(configPath, unit) else defaultValue, unit))
  }

  def valueWithDefault[T](path: String, default: T)(extractValue: (String) => T): T = option(path)(extractValue).getOrElse(default)
}

trait OptionConfigurations {
  val config: Config
  
  def optionString(path: String) = option(path)(config.getString)
  
  def option[T](path: String)(extractValue: (String) => T): Option[T] = if (config.hasPath(path)) Option(extractValue(path)) else None
}
