package com.xebialabs.xlplatform.settings

import java.io.File
import java.util.concurrent.TimeUnit

import akka.actor.{ExtendedActorSystem, ExtensionIdProvider, ExtensionId, Extension}
import akka.util.Timeout
import com.typesafe.config.Config
import scala.concurrent.duration._

object CommonSettings extends ExtensionId[CommonSettings] with ExtensionIdProvider {

  override def createExtension(system: ExtendedActorSystem): CommonSettings = new CommonSettings(system.settings.config)

  override def lookup(): ExtensionId[_ <: Extension] = CommonSettings
}

class CommonSettings(config: Config) extends Extension {
  lazy val tasker = new TaskerSettings(config)
  lazy val satellite = new SatelliteSettings(config)
}

class TaskerSettings(val config: Config) extends DefaultConfigurations {
  lazy val recoveryDir = new File(config.getString("deployit.task.recovery-dir"))
  lazy val tickDuration = config.getDuration("akka.scheduler.tick-duration", TimeUnit.MILLISECONDS)
  lazy val askTimeout = timeout("deployit.tasker.askTimeout", 1 * 1000, TimeUnit.MILLISECONDS)
}

class SatelliteSettings(val config: Config) extends DefaultConfigurations {
  lazy val uploadIdleTimeout: Timeout = timeout("satellite.timeout.upload.idle", 10 * 1000, TimeUnit.MILLISECONDS)
  lazy val pingTimeout = timeout("satellite.timeout.ping", 5 * 1000, TimeUnit.MILLISECONDS)
  lazy val remoteAskTimeout = timeout("satellite.timeout.remoteAsk", 60 * 1000, TimeUnit.MILLISECONDS)
}

trait DefaultConfigurations {
  val config: Config

  def timeout(configPath: String, defaultValue: Long, unit: TimeUnit) = {
    Timeout(FiniteDuration(if (config.hasPath(configPath)) config.getDuration(configPath, unit) else defaultValue, unit))
  }
}
