package com.xebialabs.satellite.engine

import com.xebialabs.deployit.engine.tasker.RecoverySupervisorActor.{Delete, ReadAll, Tasks, Write}
import com.xebialabs.deployit.engine.tasker._
import com.xebialabs.deployit.engine.tasker.log.DefaultStepLogFactory
import org.apache.pekko.actor.{ActorLogging, ActorRef, Props, Stash}

object EngineSupervisor {
  def props(recoverySupervisor: ActorRef) = Props(classOf[EngineSupervisor], recoverySupervisor)
  val name = "engines"

  case class CleanUp(taskId: TaskId)

  case class GetOrCreateEngine(task: Task)
  case class Engine(engine: ActorRef)

  case object GetEngines
  case class Engines(engines: Map[String, ActorRef])
}

class EngineSupervisor(val recoverySupervisor: ActorRef) extends BaseExecutionActor with ActorLogging with Stash {

  import EngineSupervisor._

  private val stepLogFactory = new DefaultStepLogFactory

  override def preStart(): Unit = {
    super.preStart()
    recoverySupervisor ! ReadAll
    context become recover
  }

  override def receive: Receive = {
    case GetOrCreateEngine(task) => sender() ! Engine(findOrCreate(task))
    case GetEngines => sender() ! Engines(context.children.map { ref => (ref.path.name, ref) }.toMap)
    case CleanUp(taskId) => recoverySupervisor ! Delete(taskId)
  }

  def findOrCreate(task: Task): ActorRef = {
    context.child(task.getId()).getOrElse {
      recoverySupervisor ! Write(task)
      create(task)
    }
  }

  def create(task: Task): ActorRef = {
    createChild(BlockExecutionEngine.props(task), task.getId())
  }

  def recover: Receive = {
    case Tasks(tasks) =>
      for (task <- tasks) {
        log.info(s"recovering task specification for task : '${task.getId()}'")
        task.context.stepLogFactory = stepLogFactory
        task.context.childContexts.foreach(c => c._2.stepLogFactory = stepLogFactory)
        create(task)
      }
      unstashAll()
      context.unbecome()
    case _ =>
      stash()
  }

}
