package com.xebialabs.satellite.engine


import com.xebialabs.deployit.engine.api.execution.BlockExecutionState
import com.xebialabs.deployit.engine.tasker.BlockExecutingActor.{BlockDone, BlockStateChanged}
import com.xebialabs.deployit.engine.tasker.StateChangeEventListenerActor.{StepStateEvent, TaskStateEvent}
import com.xebialabs.deployit.engine.tasker._
import com.xebialabs.deployit.engine.tasker.distribution.MaxSizeMessageSequenceSender
import com.xebialabs.deployit.engine.tasker.messages._
import com.xebialabs.satellite.engine.BlockExecutionEngine.{CurrentBlockState, GetBlockState}
import org.apache.pekko.actor.{ActorRef, Props, Stash, Terminated}
import org.apache.pekko.event.EventStream

import scala.jdk.CollectionConverters._

object BlockExecutionEngine {
  def props(task: Task): Props = Props(new BlockExecutionEngine(task))

  case class GetBlockState(taskId: TaskId, blockPath: Option[BlockPath] = None)

  case class CurrentBlockState(taskId: TaskId, blocks: Seq[Block])
}

class BlockExecutionEngine(task: Task) extends BaseExecutionActor with Stash with HandleCurrentBlockState {

  val eventStream: EventStream = context.system.eventStream

  val taskId: String = task.getId()

  var stateListener: ActorRef = _

  private val stateListenerName = "state-listener"

  override def preStart(): Unit = {
    registerStateListeners()
  }

  override def receive: Receive = handleExecuteMessages orElse handleStateMessages orElse handleChildTermination

  def handleExecuteMessages: Receive = ReceiveWithMdc(task) {
    case start@StartBlock(`taskId`, blockPath, _) =>
      val blockActor = createOrLookUpBlockExecutionActor(blockPath)
      logger.debug(s"Forwarding $start to $blockActor")
      blockActor.foreach(_ forward start)

    case StopBlock(`taskId`, blockPath) =>
      forwardToChildBlocks(blockPath)(b => StopBlock(taskId, b.id))

    case AbortBlock(`taskId`, blockPath) =>
      forwardToChildBlocks(blockPath)(b => AbortBlock(taskId, b.id))

    case env@BlockEnvelope(`taskId`, blockPath, msg) =>
      val blockActor = createOrLookUpBlockExecutionActor(blockPath)
      logger.debug(s"Forwarding content of the envelope $env to $blockActor")
      blockActor.foreach(_ forward msg)

    case RemoteTaskStateEvent(`taskId`, prevState, currState) =>
      logger.debug(s"Sending TaskStateEvent[$prevState,$currState] to stateListener")
      stateListener ! TaskStateEvent(taskId, task, prevState, currState)

    case TaskStateEventHandled(`taskId`, _, currState) if currState.isFinal =>
      logger.info(s"Stopping engine because task finished with state $currState")
      context stop self
  }

  def handleStateMessages: Receive = {
    case msg@GetBlockState(`taskId`, None) =>
      val origSender = sender()
      val children = context.children.filterNot(child => Seq(stateListenerName).contains(child.path.name))
      children.foreach(_ ! msg)
      implicit val excludeTerminated: ExcludeOnTerminate = (msg, child) => msg.blocks.forall(_.getId() == child.path.name)
      implicit val whenDone: WhenDone = { states =>
        debug(s"All states received: $states, sending to $origSender")
        origSender ! CurrentBlockState(taskId, states.flatMap(_.blocks))
      }
      waitForBlockStatusesOrFinish(expectedResponses = children.size)
    case GetBlockState(`taskId`, Some(path)) =>
      sender() ! CurrentBlockState(taskId, task.getBlock(path).toList)
  }

  def handleChildTermination: Receive = {
    case Terminated(child) => debug(s"Child $child terminated.")
  }

  private def createOrLookUpBlockExecutionActor(blockPath: BlockPath) = {
    if (task.getBlock(blockPath).isDefined) {
      Option(context.child(blockPath.toBlockId).getOrElse {
        val child = createChild(BlockExecution.props(task, blockPath), blockPath.toBlockId)
        context.watch(child)
        child
      })
    } else {
      sender() ! PathsNotFound(Seq(blockPath))
      None
    }
  }

  private def forwardToChildBlocks(blockPath: BlockPath)(msg: ExecutableBlock => AnyRef) = {
    childBlocksMatching(blockPath).collect {
      case block: StepBlock => forwardToChild(block.id, msg(block))
      case block: CompositeBlock => block.blocks.filter(_ != null).foreach(c => forwardToChild(c.id, msg(c)))
    }
  }

  private def childBlocksMatching(blockPath: BlockPath) = {
    task.block.getPhases.asScala.map(_.block).filter(b => b != null && blockPath.isChild(b.id))
  }

  private def forwardToChild(blockPath: BlockPath, msg: AnyRef): Unit = {
    val child: Option[ActorRef] = context.child(blockPath.toBlockId)
    logger.debug(s"forwarding $msg to $child")
    child.foreach(_ forward msg)
  }

  private def registerStateListeners(): Unit = {
    stateListener = createChild(StateChangeEventListenerActor.props(task.getId(), self), stateListenerName)
    eventStream.subscribe(stateListener, classOf[StepStateEvent])
  }
}

object BlockExecution {
  def props(task: Task, blockPath: BlockPath): Props = Props(new BlockExecution(task, blockPath))
}

class BlockExecution(task: Task, blockPath: BlockPath) extends BaseExecutionActor with ModifyStepsSupport with MaxSizeMessageSequenceSender {

  override val shouldNotChunk: Boolean = false

  val taskId: String = task.getId()

  val block: ExecutableBlock = task.getBlock(blockPath).get.asInstanceOf[ExecutableBlock]

  val blockId: BlockPath = block.id

  def receive: Receive = handleCommand

  def handleCommand: Receive = forwardStepModification(taskId, createOrLookUpBlockActor()) orElse {
    case StartBlock(`taskId`, `blockId`, runMode) =>
      info(s"Received [Start] message for task [$taskId]")
      createOrLookUpBlockActor() ! Start(taskId, runMode)
      context become (handleCommand orElse notifyMaster(sender()))

    case msg@StopBlock(`taskId`, `blockId`) =>
      debug(s"[${block.id}] : Received [$msg], stopping sub-block")
      context.child(blockId.toBlockId).foreach(_ ! Stop(taskId))
      context become (handleCommand orElse notifyMaster(sender()))

    case msg@AbortBlock(`taskId`, `blockId`) =>
      debug(s"[${block.id}] : Received [$msg] message, aborting sub-block")
      context.child(blockId.toBlockId).foreach(_ ! Abort(taskId))
      context become (handleCommand orElse notifyMaster(sender()))

    case GetBlockState(`taskId`, _) =>
      sender() ! CurrentBlockState(taskId, Seq(block))

    case ReportBlockState =>
      sendChunked(sender(), BlockStateReport(taskId, block.getId(), block.getState()))
  }

  def createOrLookUpBlockActor(): ActorRef = {
    context.child(blockId.toBlockId) match {
      case Some(ref) => ref
      case None => block match {
        case sb: StepBlock => createChild(StepBlockExecutingActor.props(task, sb, task.getContext), sb.id.toBlockId)
        case cb: CompositeBlock => createChild(BlockExecutingActor.props(task, cb, task.getContext, SatelliteBlockRouter.Satellite), cb.id.toBlockId)
      }
    }
  }

  def notifyMaster(master: ActorRef): Receive = {
    case blockState@BlockStateChanged(`taskId`, `block`, _, _) =>
      info(s"[${block.id}] : Sending BlockStateChange(${blockState.oldState}->${blockState.state}) message for [${block.id}]")
      sendChunked(master, blockState)

    case BlockDone(`taskId`, `block`) =>
      debug(s"[${block.id}] : Received BlockDone message for [${block.id}]")

      if (block.getState() == BlockExecutionState.DONE) {
        harakiri(s"[${block.id}] : All is [EXECUTED]")
      } else {
        context become receive
      }

      info(s"[${block.id}] : All sub blocks of [${block.id}] completed with state ${block.getState()}, notifying parent")
      sendChunked(master, BlockDone(taskId, block))
  }
}
