package com.xebialabs.deployit.plugin.satellite

import java.io.File
import java.net.URI
import java.nio.file

import akka.actor.{ActorRef, ActorSystem}
import akka.pattern._
import akka.util.Timeout
import com.xebialabs.deployit.engine.tasker.satellite.ActorLocator
import com.xebialabs.deployit.engine.tasker.{TaskExecutionContext, TaskSpecification}
import com.xebialabs.deployit.plugin.api.flow.{ExecutionContext, Step, StepExitCode}
import com.xebialabs.deployit.plugin.satellite.UploadTaskSupervisor.Protocol.{FileToUpload, UploadConfig, UploadFilesTask, UploadTaskFinished}
import com.xebialabs.deployit.plugin.satellite.extension.{ExtensionsLocator, FileSystemExtensionsLocator}
import com.xebialabs.satellite.future.AwaitForever
import com.xebialabs.satellite.protocol.{Directories, Paths, ReplaceExtensionsFromTask}
import com.xebialabs.xlplatform.satellite.Satellite

import scala.beans.BeanProperty
import scala.concurrent.ExecutionContextExecutor
import scala.concurrent.duration._

case class SyncExtensionsStep(actorLocator: ActorLocator, satelliteAddress: SatelliteAddress,
                              @BeanProperty description: String, extensionsLocator: ExtensionsLocator,
                              @transient system: ActorSystem, @transient _uploader: ActorRef)
  extends SatelliteActorSystemStep(system) with AwaitForever with SatellitePluginsSugar {

  @transient lazy val uploader: ActorRef = if (_uploader != null) _uploader else SatelliteActors.uploader

  override def execute(ctx: ExecutionContext): StepExitCode = {
    ctx.logOutput(s"Connecting to satellite at $satelliteAddress")

    ctx.logOutput(s"Calculating which plugins need to be updated")
    val deltas = calculateExtensionsDelta(extensionsLocator, ctx, satelliteCommunicatorSystem)
    sendFilesToSatellite(deltas, ctx) match {
      case None =>
        ctx.logOutput("Satellite is up-to-date.")
        StepExitCode.SUCCESS
      case Some(UploadTaskFinished(_, Some(error))) =>
        ctx.logError(s"Error while sending extensions to the satellite. ($error)")
        StepExitCode.FAIL
      case Some(UploadTaskFinished(filesUploaded, None)) =>
        replaceExtensionsIfNeeded(filesUploaded, ctx)
    }
  }

  private def sendFilesToSatellite(deltas: InstalledExtensionsDelta, ctx: ExecutionContext) = {
    if (!deltas.empty) {
      val directories = List(
        Directories.PLUGINS -> deltas.plugins,
        Directories.PLUGIN_HOTFIXES -> deltas.pluginHotfix,
        Directories.SATELLITE_HOTFIXES -> deltas.satelliteHotfix,
        Directories.EXT -> deltas.ext
      )

      logIfDirNeedsToBeSynced(directories, ctx)
      val files = filesToUpload(directories)
      val uploadConfig = generateUploadConfig(ctx)
      val uploadTask = UploadFilesTask(uploadConfig, files)

      implicit val timeout = Timeout(1.day)
      implicit val executionContext = satelliteCommunicatorSystem.dispatcher

      Some(blockOrThrow((uploader ? uploadTask).mapTo[UploadTaskFinished]))
    } else None
  }

  private def logIfDirNeedsToBeSynced(directories: List[(String, Set[URI])], ctx: ExecutionContext) {
    directories.foreach { case (dir, uris) =>
      if (uris.isEmpty) {
        ctx.logOutput(s"$dir is up-to-date")
      } else {
        ctx.logOutput(s"$dir needs to be synced")
      }
    }
  }

  private def filesToUpload(directories: List[(String, Set[URI])]) = {
    directories.foldLeft(List[FileToUpload]()) { case (accumulator, (_, uris)) =>
      accumulator ::: uris.map { case uri =>
        val pluginFile = PluginLocalFile(new File(uri))

        val basePath = file.Paths.get(new java.io.File(".").getCanonicalPath)
        val filePath = file.Paths.get(pluginFile.getPath)
        val relativePath = basePath.relativize(filePath).getParent
        FileToUpload(relativePath.toString.replace(File.separator, "/"), pluginFile)
      }.toList
    }
  }

  private def generateUploadConfig(ctx: ExecutionContext): UploadConfig = {
    val task = ctx.getAttribute(TaskExecutionContext.CACHE_KEY).asInstanceOf[TaskSpecification]
    UploadConfig(task.getId, ctx, actorLocator, satelliteAddress)
  }

  private def replaceExtensionsIfNeeded(filesUploaded: Long, ctx: ExecutionContext) = {
    if (filesUploaded > 0) {
      ctx.logOutput("Replacing existing satellite extensions with the uploaded ones. Satellite will be restarted.")
      ctx.setAttribute("restartRequired", true)
      implicit val timeout: Timeout = Timeout(1.day)
      implicit val executionContext: ExecutionContextExecutor = satelliteCommunicatorSystem.dispatcher
      val task = ctx.getAttribute(TaskExecutionContext.CACHE_KEY).asInstanceOf[TaskSpecification]
      val satellitePluginManagement = actorLocator.locate(Paths.pluginManagement)(satelliteCommunicatorSystem)
      blockOrThrow(satellitePluginManagement ? ReplaceExtensionsFromTask(task.getId))
    } else {
      ctx.logOutput("Replacing extensions not necessary")
    }
    StepExitCode.SUCCESS
  }

  @BeanProperty val order = Step.DEFAULT_ORDER
}

object SyncExtensionsStep {

  def apply(satellite: Satellite)(implicit satelliteCommunicatorSystem: ActorSystem, uploader: ActorRef): SyncExtensionsStep = {
    new SyncExtensionsStep(
      ActorLocator(satellite), SatelliteAddress(satellite),
      s"Synchronise extensions on satellite ${satellite.getName}",
      FileSystemExtensionsLocator, satelliteCommunicatorSystem, uploader
    )
  }

}
