package com.xebialabs.deployit.plugin.satellite

import com.xebialabs.deployit.plugin.api.udm.Parameters
import com.xebialabs.xlplatform.satellite.{Satellite, SatelliteGroup}
import java.util.{List => JList}

import akka.actor.{ActorRef, ActorSystem}

import collection.convert.ImplicitConversions._
import com.xebialabs.deployit.plugin.api.flow.Step

object SatelliteTasksHelper {

  def ping(satellite: Satellite): JList[Step] = {
    implicit val satelliteCommunicatorSystem = SatelliteCommunicatorSystem.actorSystem
    satellite match {
      case group: SatelliteGroup =>
        group.getSatellites.toList.map(pingSat)
      case sat: Satellite =>
        pingSat(sat) :: Nil
    }
  }

  def sync(satellite: Satellite, parameters: Parameters): JList[Step] = {
    implicit val satelliteCommunicatorSystem = SatelliteCommunicatorSystem.actorSystem
    implicit val uploader = SatelliteCommunicatorSystem.uploader

    satellite match {
      case group: SatelliteGroup =>
        val sats = group.getSatellites.toList
        sats.flatMap(syncRestart) ::: sats.map(waitFor(_, parameters))
      case sat: Satellite =>
        syncRestart(sat) ::: (waitFor(sat, parameters) :: Nil)
    }
  }

  def restartSatellite(satellite: Satellite, parameters: Parameters): JList[Step] = {
    implicit val satelliteCommunicatorSystem = SatelliteCommunicatorSystem.actorSystem
    satellite match {
      case group: SatelliteGroup =>
        val sats = group.getSatellites.toList
        sats.flatMap(restart(_, parameters)) ::: sats.map(waitFor(_, parameters))
      case sat: Satellite =>
        restart(sat, parameters) ::: (waitFor(sat, parameters) :: Nil)
    }
  }

  private[this] def syncRestart(sat: Satellite)(implicit satelliteCommunicatorSystem: ActorSystem, uploader: ActorRef): List[Step] = {
    SyncExtensionsStep(sat) :: SoftRestartSatelliteStep(sat) :: Nil
  }

  private[this] def restart(sat: Satellite, parameters: Parameters)(implicit satelliteCommunicatorSystem: ActorSystem): List[Step] =
    if (parameters.getProperty("force")) {
      ForceRestartSatelliteStep(sat) :: Nil
    } else {
      SoftRestartSatelliteStep(sat) :: Nil
    }

  private[this] def waitFor(sat: Satellite, parameters: Parameters)(implicit satelliteCommunicatorSystem: ActorSystem): Step = {
    WaitingForRestartStep(sat, parameters.getProperty("maxAttempts"), parameters.getProperty("delay"))
  }

  private[this] def pingSat(sat: Satellite)(implicit sateCommunicatorSystem: ActorSystem): Step = PingSatelliteStep(sat)

}
