package com.xebialabs.deployit.plugin.satellite

import akka.actor._
import akka.util.Timeout
import com.xebialabs.deployit.engine.tasker.satellite.ActorLocator
import com.xebialabs.deployit.plugin.api.flow.ExecutionContext
import com.xebialabs.deployit.plugin.satellite.Pinger.{Unavailable, Up}
import com.xebialabs.satellite.protocol.{Paths, Ping, PingReply}
import com.xebialabs.xlplatform.satellite.Satellite
import com.xebialabs.xlplatform.settings.CommonSettings

import scala.concurrent.{Await, ExecutionContextExecutor}
import scala.concurrent.duration.FiniteDuration
import scala.concurrent.duration._

object Pinger {

  val FIVE_PINGS = 5

  def defaultPingTimeout(satelliteCommunicatorSystem: ActorSystem): FiniteDuration =
    CommonSettings(satelliteCommunicatorSystem).satellite.pingTimeout.duration

  def isUp(satellite: Satellite, noPings: Int = FIVE_PINGS)
          (implicit ctx: ExecutionContext, satelliteCommunicatorSystem: ActorSystem): Boolean =
    isUp(SatelliteAddress(satellite), ActorLocator(satellite), defaultPingTimeout(satelliteCommunicatorSystem), noPings, Clock())

  def isUp(satelliteAddress: SatelliteAddress, actorLocator: ActorLocator, pingTimeout: FiniteDuration,
           pingCount: Int, clock: Clock)
          (implicit ctx: ExecutionContext, satelliteCommunicatorSystem: ActorSystem): Boolean =
  {
    val pinger = satelliteCommunicatorSystem.actorOf(Pinger.props(ctx, pingTimeout, satelliteAddress, clock, pingCount))
    ctx.logOutput(s"Connecting to satellite at $satelliteAddress")

    val grace = 100.millis
    import akka.pattern._
    implicit val timeout: Timeout = Timeout(pingTimeout + grace) // allow Pinger to send back a result even after timeout
    implicit val dispatcher: ExecutionContextExecutor = satelliteCommunicatorSystem.dispatcher

    Await.result(
      (pinger ? Pinger.Start(actorLocator.locate(Paths.ping))).mapTo[PingResult]
      .recover { case error =>
          ctx.logError(s"Operation failed  (${error.getMessage})")
          Unavailable
      }.map(Up == _), timeout.duration + grace) // more grace time allow local handling even after timeout
  }

  def props(ctx: ExecutionContext, timeout: FiniteDuration, satelliteAddress: SatelliteAddress, clock: Clock = Clock(),
            noPings: Int = Pinger.FIVE_PINGS) =
    Props(new Pinger(ctx, timeout, satelliteAddress, clock, noPings))

  case class Start(target: ActorSelection)

  sealed trait PingResult

  case object Up extends PingResult

  case object Unavailable extends PingResult

}

class Pinger(ctx: ExecutionContext, pingTimeout: FiniteDuration, satelliteAddress: SatelliteAddress, clock: Clock, pingCount: Int) extends Actor {

  def receive: Receive = waitingToStart

  def waitingToStart: Receive = {
    case Pinger.Start(target) =>
      if (!CommonSettings(context.system).satellite.enabled) {
        ctx.logError("Satellite has not been enabled on the XL Deploy server. Please ensure 'satellite.enabled = true' is set in the system.conf file of XL Deploy")
        sender() ! Unavailable
        context.stop(self)
      } else {
        target ! Ping

        ctx.logOutput(s"Waiting for connection to satellite at $satelliteAddress")
        context.setReceiveTimeout(pingTimeout)

        continueWith(sender()) {
          loggingPing(target, clock.currentTimeMillis(), pingCount, sender())
        }
      }
  }

  def loggingPing(target: ActorSelection, startTime: Long, remainingPingsToReceive: Int, requester: ActorRef): Receive = {
    case PingReply(uptime) =>
      if (remainingPingsToReceive == pingCount) ctx.logOutput(s"Satellite is live (uptime: $uptime sec)")
      val currentTime = clock.currentTimeMillis()
      logPing(startTime)
      if (remainingPingsToReceive > 1) {
        target ! Ping
        continueWith(requester) {
          loggingPing(target, currentTime, remainingPingsToReceive - 1, requester)
        }
      } else {
        requester ! Up
        context stop self
      }
  }

  private def logPing(startTime: Long): Unit = {
    ctx.logOutput(s"Ping: ${clock.currentTimeMillis() - startTime} ms")
  }

  private def continueWith(requester: ActorRef)(nextBehavior: Receive): Unit = {
    context become (nextBehavior orElse handleTimeout(requester))
  }

  private def handleTimeout(requester: ActorRef): Receive = {
    case ReceiveTimeout =>
      ctx.logError(s"The satellite at $satelliteAddress cannot be reached. Please check whether it is running.")
      requester ! Unavailable
      context stop self
  }
}
