package com.xebialabs.deployit.plugin.satellite

import java.io.ObjectInputStream

import akka.actor.ActorSystem
import com.xebialabs.deployit.plugin.api.flow.{ExecutionContext, Step, StepExitCode}
import com.xebialabs.xlplatform.satellite.{Satellite, SatelliteGroup}

import scala.util.Random

class ChooseSatelliteInGroupStep(satelliteGroup: SatelliteGroup)
                                (implicit @transient var satelliteCommunicatorSystem: ActorSystem) extends Step {
  /**
    * The order of the step in the step list.
    *
    * @return the order.
    */
  override def getOrder: Int = 10

  /**
    * Returns a description of what this step will really do on execution.
    *
    * @return the description.
    */
  override def getDescription: String = s"Choose a satellite to execute task in $satelliteGroup"

  /**
    * @param ctx Context info required to execute the step.
    * @return exit code of the execution, successful, failed or paused.
    * @throws Exception Any type of exception can be thrown,
    *                   this to prevent implementors of having to wrap any non-runtime exceptions.
    */
  override def execute(ctx: ExecutionContext): StepExitCode = {
    import collection.convert.ImplicitConversions._
    val chosen = new RandomSatelliteChooser(ctx, satelliteCommunicatorSystem).pick(satelliteGroup.getSatellites.toList.sortBy(_.getId))
    chosen.map({ sat =>
      ctx.logOutput(s"Chose Satellite $sat for group $satelliteGroup")
      satelliteGroup.setChosenSatellite(sat)
      StepExitCode.SUCCESS
    }).getOrElse({
      ctx.logError(s"Could not choose satellite for group $satelliteGroup")
      StepExitCode.FAIL
    })
  }

  private def readObject(in: ObjectInputStream): Unit = {
    in.defaultReadObject()
    satelliteCommunicatorSystem = SatelliteCommunicatorSystem.actorSystem
  }
}

object ChooseSatelliteInGroupStep {

  def apply(satelliteGroup: SatelliteGroup)(implicit satelliteCommunicatorSystem: ActorSystem): ChooseSatelliteInGroupStep =
    new ChooseSatelliteInGroupStep(satelliteGroup)
}

trait SatelliteChooser {
  def pick(satellites: List[Satellite]): Option[Satellite]
}

class RandomSatelliteChooser(ctx: ExecutionContext, satelliteCommunicatorSystem: ActorSystem) extends SatelliteChooser {
  override def pick(satellites: List[Satellite]): Option[Satellite] = {
    val randomized = Random.shuffle(satellites)
    ctx.logOutput(s"Trying satellite group ${satellites.mkString("[", ", ", "]")} in order ${randomized.mkString(", ")}")
    randomized.find {
      case sat if Pinger.isUp(sat, 1)(ctx, satelliteCommunicatorSystem) => true
      case sat =>
        ctx.logOutput(s"Satellite $sat is not responding to pings, not selecting it.")
        false
    }
  }
}