package com.xebialabs.deployit.plugin.satellite

import akka.actor._
import com.typesafe.config.Config
import com.typesafe.config.ConfigFactory._
import com.xebialabs.xlplatform.settings.{CommonSettings, ConfigSupport, SecuritySettings}

import scala.concurrent.ExecutionContext

object SatelliteCommunicatorSystem extends ConfigSupport {

  lazy val actorSystem: ActorSystem = newActorSystem(parseResources("system.conf"))

  def newActorSystem(systemConfig: Config, systemName: Option[String] = None): ActorSystem = {
    val commonSettings = new CommonSettings(systemConfig)
    val defaultName = if (commonSettings.satellite.enabled) "satellite-task-sys" else "task-sys"
    ActorSystem.create(systemName.getOrElse(defaultName), loadConfig(systemConfig))
  }

  def loadConfig(systemConfig: Config): Config = {
    val commonSettings = new CommonSettings(systemConfig)
    val fallbacks = if (commonSettings.satellite.enabled) {
      Seq(hostAddressConfiguration("satellite.hostname"), sslRemoteTransports(commonSettings.security), parseResources("system-reference-satellite-enabled.conf"))
    } else {
      Seq(parseResources("system-reference-satellite-disabled.conf"))
    }
    loadWithFallbacks(systemConfig, fallbacks: _*)
  }

  private[this] def sslRemoteTransports(security: SecuritySettings): Config = {
    import scala.collection.JavaConverters._
    import scala.collection.convert.ImplicitConversionsToJava._
    if (security.enabled) {
      parseMap(Map("satellite.remote.enabled-transports" -> Seq("akka.remote.netty.ssl").asJava))
    } else {
      empty()
    }
  }

  lazy val uploaderPath = uploader.path

  lazy val uploader = actorSystem.actorOf(StreamingSupervisor.props(), "uploader")

  lazy val config = CommonSettings(actorSystem).satellite

  implicit lazy val dispatcher: ExecutionContext = actorSystem.dispatcher
}
