package com.xebialabs.deployit.plugin.satellite

import java.io.BufferedInputStream
import java.net.InetSocketAddress

import akka.actor._
import akka.pattern.pipe
import akka.stream.scaladsl._
import akka.stream.{ClosedShape, ActorMaterializer, ActorMaterializerSettings}
import com.xebialabs.deployit.plugin.satellite.UploadTaskSupervisor.Protocol.{FileToUpload, UploadConfig}
import com.xebialabs.satellite.protocol.UploadReply.Ready
import com.xebialabs.satellite.protocol._
import com.xebialabs.satellite.streaming.DigesterStage.Digest
import com.xebialabs.satellite.streaming.SslStreamingSupport.SslConfig
import com.xebialabs.satellite.streaming._
import com.xebialabs.xlplatform.settings.CommonSettings

import scala.concurrent.Promise
import scala.concurrent.duration.{Duration, FiniteDuration}
import scala.util.{Failure, Success, Try}

object FileUploader {
  def props(requester: ActorRef, config: UploadConfig, file: FileToUpload): Props =
    Props(new FileUploader(requester, config, file))
}

class FileUploader(requester: ActorRef, config: UploadConfig, fileToUpload: FileToUpload) extends Actor with ActorLogging {
  val settings = ActorMaterializerSettings(context.system).withDispatcher("streaming.StreamingDispatcher")
  implicit val materializer = ActorMaterializer(settings)
  implicit val system = context.system
  val securitySettings = CommonSettings(system).security
  val satelliteSettings = CommonSettings(system).satellite

  var uploadIdleTimeout: FiniteDuration = _

  override def preStart() {
    val remoteUploadActor = config.satellite.locate(com.xebialabs.deployit.engine.tasker.satellite.Paths.tasks)
    remoteUploadActor ! UploadFileForTask(fileToUpload.id, config.taskId, fileToUpload.file.getName, fileToUpload.path)
  }

  def receive = reachingSatellite

  def reachingSatellite: Receive = {
    case UploadReply.InitConnection(port, chunkSize, compression, wantTls) =>
      val fileReceiver = sender()

      implicit val streamingConfig = StreamConfig(
        chunkSize = chunkSize,
        compression = compression
      )
      val satelliteReady = Promise[Unit]()

      Try {
        SslStreamingSupport.SslConfig(wantTls, securitySettings)
      }.flatMap(sslConfig => establishConnection(port, sslConfig, satelliteReady)) match {
        case Success(futureConnection) =>
          implicit val ec = context.dispatcher
          futureConnection.pipeTo(self)
          context.setReceiveTimeout(satelliteSettings.streamingConnectionTimeout.duration)
          context become waitingForConnection(fileReceiver, satelliteReady)
        case Failure(exc) =>
          handleConnectionError(s"Cannot enable SSL connection: ${exc.getMessage}", fileReceiver, Some(exc))
      }
    case AlreadyRegistered =>
      requester ! UploadReply.Error("Task is already registered")
      context stop self
  }

  private def establishConnection(port: Int, sslConfig: SslConfig, satelliteReady: Promise[Unit])(implicit streamingConfig: StreamConfig) = {
    Try {
      val satelliteAddress = new InetSocketAddress(config.satelliteAddress.hostname, port)
      log.info(s"'connecting to $satelliteAddress' for upload '${fileToUpload.id}'")

      val fileStream = fileToUpload.file.getRawStream
      log.info("extracted stream from file to upload")
      val buffStream = new BufferedInputStream(fileStream)
      log.info("Sending file stream to  stream from file to upload")
      val byteSource = UploadStage.source(satelliteReady.future, buffStream) {
        case Digest(checksum, fileSize) => self ! StreamDone(checksum, fileSize)
      }

      val connection = Tcp().outgoingConnection(satelliteAddress)

      val wrappedConnection = SslStreamingSupport.wrapWithSsl(sslConfig.asClient.eagerClose, connection)
      val connectToDevNull = RunnableGraph.fromGraph(GraphDSL.create(byteSource, wrappedConnection)((_, c) => c) {implicit builder =>
          (bs, wrappedConn) =>
            import GraphDSL.Implicits._
            bs ~> wrappedConn ~> Sink.ignore
            ClosedShape.getInstance
      })
      connectToDevNull.run()
    }

  }

  def waitingForConnection(fileReceiver: ActorRef, satelliteReady: Promise[Unit]): Receive = {
    case connection@Tcp.OutgoingConnection(remoteAddress, localAddress) =>
      context.setReceiveTimeout(Duration.Undefined)
      log.info(s"Connected $localAddress -> $remoteAddress")
      fileReceiver ! Connected(connection.localAddress)
      log.info("Starting Sync operation...")
      context become inSync(fileReceiver, satelliteReady)
    case ReceiveTimeout =>
      handleConnectionError(s"Unable to establish connection within ${satelliteSettings.streamingConnectionTimeout.duration.toSeconds}.", fileReceiver)
  }

  private def handleConnectionError(description: String, fileReceiver: ActorRef, exception: Option[Throwable] = None) {
    log.error(description)
    exception.foreach(e => log.debug(e.toString))
    fileReceiver ! CannotConnect(description)
    requester ! UploadReply.Error(description)
    context stop self
  }

  def inSync(fileReceiver: ActorRef, sync: Promise[Unit]): Receive = {
    case Ready =>
      sync.success(Unit)
      log.info("Sync operation successful, sending file to FileReceiver")
      context become (streaming(fileReceiver) orElse handleError)
  }

  def streaming(fileReceiver: ActorRef): Receive = {
    case StreamDone(checksum, fileLength) =>
      log.info(s"Sending $fileLength bytes to upload ")
      fileReceiver ! FileUploaded(checksum, fileLength)
      config.ctx.logOutput(s"${fileToUpload.path}/${fileToUpload.file.getName}: uploaded $fileLength bytes. Checksum: $checksum")

    case UploadReply.Done =>
      log.info(s"'${fileToUpload.path}' uploaded successfully")

      requester ! UploadReply.Done
      context stop self
  }

  def handleError: Receive = {
    case error@UploadReply.Error(cause) =>
      log.warning(s"'${fileToUpload.file.getName}' failed")

      requester forward error
      context stop self
  }

  case class StreamDone(checksum: String, fileLength: Long)

}