package com.xebialabs.deployit.plugin.satellite

import akka.actor._
import com.xebialabs.deployit.plugin.api.flow.{ExecutionContext, StepExitCode}
import com.xebialabs.satellite.protocol.{Ping, PingReply}
import com.xebialabs.xlplatform.settings.CommonSettings

import scala.concurrent.duration.FiniteDuration

object Pinger {

  case class Start(target: ActorSelection)

  val FIVE_PINGS = 5

  def props(ctx: ExecutionContext, timeout: FiniteDuration, satelliteAddress: String, clock: Clock = Clock()) = Props(classOf[Pinger], ctx, timeout, satelliteAddress, clock)

}
class Pinger(ctx: ExecutionContext, timeout: FiniteDuration, satelliteAddress: String, clock: Clock = Clock()) extends Actor {

  def receive = waitingToStart

  def waitingToStart: Receive = {
    case Pinger.Start(target) =>
      if (!CommonSettings(context.system).satellite.enabled) {
        ctx.logError("Satellite has not been enabled on the XL Deploy server. Please ensure 'satellite.enabled = true' is set in the system.conf file of XL Deploy")
        sender() ! StepExitCode.FAIL
        context.stop(self)
      } else {
        ping(target)

        ctx.logOutput(s"Waiting for connection to satellite at $satelliteAddress")
        context.setReceiveTimeout(timeout)

        continueWith(sender()) {
          receivingFirstPing(target, clock.currentTimeMillis(), sender())
        }
      }
  }

  def receivingFirstPing(target: ActorSelection, startTime: Long, requester: ActorRef): Receive = {
    case PingReply(uptime) =>
      ctx.logOutput(s"Satellite is live (uptime: $uptime sec)")

      logPing(startTime)
      ping(target)

      continueWith(requester) {
        loggingPing(target, clock.currentTimeMillis(), remainingPingToReceive = Pinger.FIVE_PINGS - 1, requester)
      }

  }

  def loggingPing(target: ActorSelection, startTime: Long, remainingPingToReceive: Int, requester: ActorRef): Receive = {
    case PingReply(_) =>
      val currentTime = clock.currentTimeMillis()

      logPing(startTime)

      if (remainingPingToReceive > 1) {

        ping(target)

        continueWith(requester) {
          loggingPing(target, currentTime, remainingPingToReceive - 1, requester)
        }

      } else {

        requester ! StepExitCode.SUCCESS
        context stop self
      }
  }

  private def logPing(startTime: Long) {
    ctx.logOutput(s"Ping: ${clock.currentTimeMillis() - startTime} ms")
  }

  private def continueWith(requester: ActorRef)(nextBehavior: Receive) {
    context become (nextBehavior orElse handleTimeout(requester))
  }

  private def handleTimeout(requester: ActorRef): Receive = {
    case ReceiveTimeout =>
      ctx.logError(s"The satellite at $satelliteAddress cannot be reached. Please check whether it is running.")
      requester ! StepExitCode.FAIL
      context stop self
  }

  private def ping(target: ActorSelection) {
    target ! Ping
  }
}
