package com.xebialabs.deployit.plugin.satellite

import java.io.ObjectInputStream

import akka.actor.ActorSystem
import akka.pattern.ask
import com.xebialabs.deployit.engine.tasker
import com.xebialabs.deployit.engine.tasker.satellite.ActorLocator
import com.xebialabs.deployit.engine.tasker.{TaskExecutionContext, TaskSpecification}
import com.xebialabs.deployit.plugin.api.flow.{ExecutionContext, Step, StepExitCode}
import com.xebialabs.satellite.future.AwaitForever
import com.xebialabs.satellite.protocol.CleanUpTask
import com.xebialabs.xlplatform.satellite.Satellite
import com.xebialabs.xlplatform.settings.CommonSettings

import scala.beans.BeanProperty

case class CleanUpSatelliteStep(satellite: Satellite, actorLocator: ActorLocator, @BeanProperty description: String, cleanupCache: TaskOnSatelliteCleanupCache)
                               (implicit @transient var satelliteCommunicatorSystem: ActorSystem) extends Step with AwaitForever {

  @BeanProperty val order = Step.DEFAULT_ORDER

  override def execute(ctx: ExecutionContext): StepExitCode = {
    if (!cleanupCache.isRegistered(satellite)) {
      ctx.logOutput(s"No need to cleanup satellite at ${SatelliteAddress(satellite)} as it never received the task.")
      return StepExitCode.SUCCESS
    }

    implicit val executionContext = satelliteCommunicatorSystem.dispatcher

    val task = ctx.getAttribute(TaskExecutionContext.CACHE_KEY).asInstanceOf[TaskSpecification]

    implicit val timeout = CommonSettings(satelliteCommunicatorSystem).satellite.pingTimeout

    val remoteTaskActor = actorLocator.locate(tasker.satellite.Paths.tasks)

    ctx.logOutput(s"Sending cleanup task to satellite at ${SatelliteAddress(satellite)}")

    val complete = (remoteTaskActor ? CleanUpTask(task.getId)).map(_ => StepExitCode.SUCCESS).recover {
      case error =>
        ctx.logError(s"Cleanup task failed with error [${error.getMessage}]")
        StepExitCode.FAIL
    }

    complete.onSuccess { case _ => ctx.logOutput(s"Cleanup task finished") }

    blockOrThrow(complete)
  }

  private def readObject(in: ObjectInputStream): Unit = {
    in.defaultReadObject()
    satelliteCommunicatorSystem = SatelliteCommunicatorSystem.actorSystem
  }
}

object CleanUpSatelliteStep extends SatelliteAddress {
  def apply(satellite: Satellite, cleanupCache: TaskOnSatelliteCleanupCache)(implicit satelliteCommunicatorSystem: ActorSystem): CleanUpSatelliteStep = {
    new CleanUpSatelliteStep(satellite, ActorLocator(satellite), s"Clean up task file on satellite ${satellite.getName}", cleanupCache)
  }
}
