package com.xebialabs.deployit.plugin.satellite

import java.io.ObjectInputStream
import java.net.URI

import akka.actor.ActorSystem
import com.xebialabs.deployit.engine.tasker.satellite.ActorLocator
import com.xebialabs.deployit.plugin.api.flow.{ExecutionContext, Step, StepExitCode}
import com.xebialabs.deployit.plugin.satellite.extension.{ExtensionsLocator, FileSystemExtensionsLocator}
import com.xebialabs.satellite.future.AwaitForever
import com.xebialabs.xlplatform.satellite.Satellite

import scala.beans.BeanProperty

case class CheckExtensionsStep(actorLocator: ActorLocator, @BeanProperty description: String, extensionsLocator: ExtensionsLocator)
                          (implicit @transient var satelliteCommunicatorSystem: ActorSystem) extends Step with AwaitForever with SatellitePluginsSugar {

  @BeanProperty val order = Step.DEFAULT_ORDER

  override def execute(ctx: ExecutionContext): StepExitCode = {
    ctx.logOutput(s"Connecting to $actorLocator")

    val installedExtensions = calculateExtensionsDelta(extensionsLocator, ctx)

    printStatus(installedExtensions.plugins, "plugins", ctx)
    printStatus(installedExtensions.hotfix, "hotfix/plugins", ctx)
    printStatus(installedExtensions.ext, "ext", ctx)

    if (List(installedExtensions.plugins, installedExtensions.hotfix, installedExtensions.ext).forall(_.isEmpty)) {
      ctx.logOutput(s"Satellite is up-to-date")
      StepExitCode.SUCCESS
    } else {
      ctx.logOutput(s"Found unsynchronized extensions")
      StepExitCode.FAIL
    }

  }

  private def printStatus(opt: Set[URI], extension: String, ctx: ExecutionContext) = {
    if(opt.isEmpty) {
      ctx.logOutput(s"$extension is up-to-date")
    } else {
      ctx.logOutput(s"Need to synchronise $extension")
    }
  }

  private def readObject(in: ObjectInputStream): Unit = {
    in.defaultReadObject()
    satelliteCommunicatorSystem = SatelliteCommunicatorSystem.actorSystem
  }
}

object CheckExtensionsStep {

  def apply(satellite: Satellite)(implicit satelliteCommunicatorSystem: ActorSystem) : CheckExtensionsStep = {
    new CheckExtensionsStep(
      ActorLocator(satellite), s"Check extensions on satellite ${satellite.getName}",
      FileSystemExtensionsLocator
    )
  }

}
