package com.xebialabs.deployit.plugin.satellite

import akka.actor._
import akka.remote.{DisassociatedEvent, QuarantinedEvent}
import com.xebialabs.deployit.engine.tasker._
import com.xebialabs.deployit.engine.tasker.satellite.ActorLocator
import com.xebialabs.deployit.io.LazyLocalFile
import com.xebialabs.deployit.plugin.api.flow.ExecutionContext
import com.xebialabs.deployit.plugin.satellite.UploadTaskSupervisor.Protocol.{FileToUpload, UploadFilesTask, UploadTaskFinished}
import com.xebialabs.deployit.plugin.satellite.UploadTaskSupervisor.UploaderConstructor
import com.xebialabs.satellite.protocol.{UploadId, UploadReply}
import com.xebialabs.xlplatform.settings.CommonSettings

object UploadTaskSupervisor {
  type UploaderConstructor = (ActorRefFactory, ActorRef, FileToUpload) => ActorRef

  object Protocol {
    case class UploadFilesTask(config: UploadConfig, files: Iterable[FileToUpload])

    case class UploadConfig(taskId: TaskId, ctx: ExecutionContext, satellite: ActorLocator, satelliteHostname: String)

    case class FileToUpload(path: String, file: LazyLocalFile, id: UploadId = UploadId())

    object EncryptionType {
      val ENCRYPTED = true
      val PLAIN = false
    }

    case class UploadTaskFinished(uploadedFiles: Long, error: Option[String] = None)
  }

  def props(requester: ActorRef, task: UploadFilesTask): Props = {
    val uploaderCons: UploaderConstructor = (factory, supervisor, file) =>
      factory.actorOf(FileUploader.props(supervisor, task.config, file), file.id.toString)
    props(requester, task, uploaderCons)
  }

  def props(requester: ActorRef, task: UploadFilesTask, uploaderCons: UploaderConstructor): Props = {
    Props(new UploadTaskSupervisor(requester, task, uploaderCons))
  }
}

class UploadTaskSupervisor(requester: ActorRef, task: UploadFilesTask, createUploader: UploaderConstructor) extends Actor with ActorLogging {
  val satelliteSettings = CommonSettings(context.system).satellite

  val iterator = task.files.iterator
  val levelOfParallelism = satelliteSettings.maxConcurrentUploads
  var concurrentUploads = 0
  var uploadedFiles = 0

  @throws[Exception](classOf[Exception])
  override def preStart() {
    context.system.eventStream.subscribe(self, classOf[DisassociatedEvent])
    context.system.eventStream.subscribe(self, classOf[QuarantinedEvent])

    if (task.files.isEmpty) {
      finishedSuccessfully()
    } else {
      0.to(levelOfParallelism).foreach(_ => uploadNextFile())
    }
  }

  override def receive: Receive = {
    case UploadReply.Error(cause) =>
      requester ! UploadTaskFinished(uploadedFiles, Some(cause))
      context.stop(self)

    case msg@UploadReply.Done =>
      concurrentUploads -= 1
      uploadedFiles += 1
      uploadNextFile()

    case e:DisassociatedEvent =>
      requester ! UploadTaskFinished(uploadedFiles, Some("The satellite seems to be unreachable at the moment. Please try again later."))
      context.stop(self)

    case e:QuarantinedEvent =>
      requester ! UploadTaskFinished(uploadedFiles, Some("The satellite seems to be permanently unreachable. Please restart the satellite and retry."))
      context.stop(self)
  }

  private def uploadNextFile() {
    if (iterator.hasNext && concurrentUploads < levelOfParallelism) {
      val file = iterator.next()
      createUploader(context, self, file)
      concurrentUploads += 1
    }
    if (concurrentUploads == 0) {
      finishedSuccessfully()
    }
  }

  private def finishedSuccessfully() {
    requester ! UploadTaskFinished(uploadedFiles)
    context.stop(self)
  }
}